<?php

namespace Drupal\unep_legislation_indigo\Drupal;

use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Symfony\Component\String\Slugger\AsciiSlugger;

class ImportUtil {
  private static array $indigo2DrupalLangCode = [
    'ara' => 'ar',
    'eng' => 'en',
    'fra' => 'fr',
    'rus' => 'ru',
    'spa' => 'es',
    'zho' => 'zh-hans',

    'bul' => 'bg',
    'cat' => 'ca',
    'dan' => 'da',
    'deu' => 'de',
    'ell' => 'el',
    'hun' => 'hu',
    'isl' => 'is',
    'jpn' => 'ja',
    'kor' => 'ko',
    'nld' => 'nl',
    'por' => 'pt',
    'sqi' => 'sq',
    'srp' => 'sr',
    'swe' => 'sv',
    'nor' => 'nn',
    'swa' => 'sz',
    'afr' => 'af',
    'zul' => 'zu',
    'xho' => 'xh',
  ];

  /**
   * @throws \Exception
   */
  public static function getLanguageCode($code): ?string {
    if (isset(self::$indigo2DrupalLangCode[$code])) {
      return self::$indigo2DrupalLangCode[$code];
    } else {
      throw new \Exception(sprintf('Failed to decode language: %s', $code));
    }
  }

  public static function getCountryByName($name, $storage = 'taxonomy_term'): ?\Drupal\Core\Entity\EntityInterface {
    if (empty($name)) {
      return NULL;
    }
    $mapping = [
      'great britain' => 'United Kingdom of Great Britain and Northern Ireland',
      'united kingdom' => 'United Kingdom of Great Britain and Northern Ireland',
    ];
    $country_name = strtolower($name);
    $country_name = $mapping[$country_name] ?? $country_name;

    $query = \Drupal::entityTypeManager()->getStorage('taxonomy_term')->getQuery();
    $query->condition('vid', 'countries')->accessCheck(FALSE);
    $or = $query->orConditionGroup()
      ->condition('name', $country_name, '=' )
      ->condition('field_iso2', $country_name, '=')
      ->condition('field_iso3', $country_name, '=');
    $query->condition($or);
    $results = $query->execute();
    if (!empty($results)) {
      return \Drupal::entityTypeManager()->getStorage($storage)->load(reset($results));
    }
    return NULL;
  }

  public static function createFilenameFromFrBrUri($uri, $extension): ?string {
    if (preg_match('/\/akn\//', $uri)) {
      $uri = preg_replace('/.*\/akn\/[a-z\-0-9]*/i', '', $uri);
    }
    if (preg_match("/\.$extension$/i", $uri)) {
      $uri = preg_replace("/\.$extension$/i", '', $uri);
    }
    $slugger = new AsciiSlugger();
    $filename = $slugger->slug($uri);
    return !empty($extension) ? $filename . '.' . $extension : $filename;
  }

  public static function formatLegislationNodeLink(NodeInterface $revision) {
    if ($revision->isLatestRevision()) {
      return Url::fromRoute('entity.node.canonical', ['node' => $revision->id()])->toString();
    } else {
      return Url::fromRoute('entity.node.revision', ['node' => $revision->id(), 'node_revision' => $revision->getRevisionId()])->toString();
    }
  }

  /**
   * Searches for a type of work.
   *
   * @param string $name
   *   The type name.
   * @param string $machineName
   *   The type machine name.
   * @param string $entityTypeId
   *   The entity type id.
   * @param string $vid
   *   The vocabulary id.
   * @param string $fieldName
   *   The field name.
   */
  public static function getTypeOfWork(string $name, string $machineName, string $entityTypeId, string $vid, string $fieldName) {
    $storage = \Drupal::entityTypeManager()->getStorage($entityTypeId);
    $query = $storage->getQuery()->accessCheck(FALSE);
    $query->condition('vid', $vid);
    $query->condition($fieldName, $machineName);
    $results = $query->execute();
    if (!empty($results)) {
      return $storage->load(reset($results));
    }
    \Drupal::logger('unep_legislation_indigo')->debug('No Type of text found for %name, %subtype.', ['%name' => $name, '%subtype' => $machineName]);
    return NULL;
  }

}
