((Drupal, once) => {
  Drupal.behaviors.softHyphensHighlighter = {
    attach(context) {
      // Track last focused input.
      let lastField = false;
      once('softHyphensFocusTracker', 'form', context).forEach((body) => {
        body.addEventListener(
          'focus',
          (e) => {
            const t = e.target;
            if (t.tagName === 'INPUT' || t.tagName === 'TEXTAREA') {
              lastField = t;
            }
          },
          true,
        );
      });

      // Mirror and marker logic.
      function getMirror(el) {
        if (!el._sh_mirror) {
          const m = document.createElement('div');
          m.className = 'sh-mirror';

          const parent =
            el.closest('.form-item-settings-label') || el.parentNode;
          parent.style.position = parent.style.position || 'relative';
          parent.appendChild(m);

          el._sh_mirror = m;
        }
        return el._sh_mirror;
      }

      function clearMarkers(el) {
        (el._sh_markers || []).forEach((m) => m.remove());
        el._sh_markers = [];
      }

      function applyMirror(el, mirror) {
        const cs = getComputedStyle(el);

        // Position mirror absolutely inside the parent
        mirror.style.position = 'absolute';
        mirror.style.top = `${el.offsetTop}px`;
        mirror.style.left = `${el.offsetLeft}px`;
        mirror.style.width = `${el.offsetWidth}px`;
        mirror.style.height = `${el.offsetHeight}px`;

        // Copy styles
        Object.assign(mirror.style, {
          boxSizing: cs.boxSizing,
          paddingTop: cs.paddingTop,
          paddingRight: cs.paddingRight,
          paddingBottom: cs.paddingBottom,
          paddingLeft: cs.paddingLeft,
          borderTopWidth: cs.borderTopWidth,
          borderRightWidth: cs.borderRightWidth,
          borderBottomWidth: cs.borderBottomWidth,
          borderLeftWidth: cs.borderLeftWidth,
          font: cs.font,
          lineHeight: cs.lineHeight,
          letterSpacing: cs.letterSpacing,
          wordSpacing: cs.wordSpacing,
          whiteSpace: 'pre-wrap',
          overflowWrap: 'break-word',
          visibility: 'hidden',
        });
      }

      function markSoftHyphens(el) {
        clearMarkers(el);
        const mirror = getMirror(el);
        applyMirror(el, mirror);

        // Safely encode text and mark soft hyphens
        mirror.innerHTML = el.value
          .split('')
          .map((char) => {
            if (char === '\u00AD') {
              return '<span data-shy>&shy;</span>';
            }
            const entityMap = {
              '<': '&lt;',
              '>': '&gt;',
              '&': '&amp;',
            };

            return entityMap[char] || char;
          })
          .join('');

        const shySpans = mirror.querySelectorAll('span[data-shy]');
        if (!shySpans.length) {
          return;
        }

        const parent = el.closest('.form-item-settings-label') || el.parentNode;
        el._sh_markers = [];

        shySpans.forEach((span) => {
          const rect = span.getBoundingClientRect();
          const parentRect = parent.getBoundingClientRect();

          const marker = document.createElement('div');
          marker.className = 'sh-marker';
          marker.style.top = `${rect.top - parentRect.top}px`;
          marker.style.left = `${rect.left - parentRect.left}px`;
          marker.style.height = `${rect.height}px`;
          marker.style.position = 'absolute';

          parent.appendChild(marker);
          el._sh_markers.push(marker);
        });
      }

      function initField(el) {
        if (el._sh_initialized) {
          return;
        }
        el._sh_initialized = true;
        const update = () => markSoftHyphens(el);
        el.addEventListener('input', update);
        el.addEventListener('scroll', update);
        window.addEventListener('resize', update);
        if ('ResizeObserver' in window) {
          const ro = new ResizeObserver(update);
          ro.observe(el);
          el._sh_resizeObserver = ro;
        }
        update();
      }

      // Apply to all matching fields in the current context.
      once(
        'softHyphensInput',
        'input[name="settings[label]"], input[name="title[0][value]"]',
        context,
      ).forEach(initField);

      // Optional: bind a button (if available).
      once('softHyphensButton', '#insert-sh', context).forEach((button) => {
        button.addEventListener('click', () => {
          if (!lastField) {
            return alert('Please focus an input or textarea first.');
          }
          const pos = lastField.selectionStart;
          lastField.value = `${lastField.value.slice(
            0,
            pos,
          )}\u00AD${lastField.value.slice(lastField.selectionEnd)}`;
          lastField.selectionStart = pos + 1;
          lastField.selectionEnd = pos + 1;
          markSoftHyphens(lastField);
          lastField.focus();
        });
      });
    },
  };
})(Drupal, once);
