<?php

namespace Drupal\unified_date\Hook;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\node\NodeInterface;
use Drupal\unified_date\UnifiedDateManager;

/**
 * Hook implementations for entities.
 */
class EntityHook {

  use StringTranslationTrait;

  /**
   * Constructs the Entity Hook class.
   */
  public function __construct(protected UnifiedDateManager $unifiedDateManager) {
  }

  /**
   * Implements hook_entity_base_field_info().
   */
  #[Hook('entity_base_field_info')]
  public function entityBaseFieldInfo(EntityTypeInterface $entity_type): array {
    $fields = [];

    // Whether the hook implementation was called for our group type.
    if ($entity_type->id() == 'node') {

      // Create a base field for the unified date.
      $fields['unified_date'] = BaseFieldDefinition::create('timestamp')
        ->setLabel($this->t('Unified date'))
        ->setDescription($this->t('The unified date field.'))
        ->setRevisionable(TRUE)
        ->setTranslatable(TRUE)
        ->setSettings([
          'max_length' => 11,
        ]);
    }

    return $fields;
  }

  /**
   * Implements hook_entity_base_field_info().
   */
  #[Hook('node_presave')]
  public function nodePresave(EntityInterface $entity): void {
    if ($entity instanceof NodeInterface) {
      // Get the date only, don't save. Entity will be saved in the subsequent
      // lifecycle of the entity.
      $entity->set('unified_date', $this->unifiedDateManager->getUnifiedDate($entity));
    }
  }

}
