<?php
/**
 * Created by PhpStorm.
 * User: steve
 * Date: 20/07/18
 * Time: 14:23
 */

namespace Drupal\universal_file_utils\Event;

use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\FileInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\Event;

abstract class UniversalFileBaseEvent extends Event {

  use StringTranslationTrait;
  use LoggerChannelTrait;

  const NAME = 'universal_file_utils.abstract';

  /**
   * Create and dispatch the relevant activity event.
   *
   * @param FileInterface $file
   *
   * @param FileUsageInterface $fileUsage
   *
   * @return UniversalFileBaseEvent
   */
  static public function Dispatch(FileInterface $file, FileUsageInterface $fileUsage, array $extras = []) {
    $usage_list = $fileUsage->listUsage($file);

    $event = new static($file, $usage_list);

    $event->extras = $event->getDefaults($extras);

    return static::doDispatch(static::NAME, $event);
  }

  /**
   * @param string $name
   * @param UniversalFileBaseEvent $event
   *
   * @return UniversalFileBaseEvent
   */
  static protected function doDispatch($name, UniversalFileBaseEvent $event) {
    static::slogger()->info(new TranslatableMarkup('Dispatching event: %x', ['%x' => $name]));
    \Drupal::service('event_dispatcher')->dispatch($name, $event);

    return $event;
  }

  //---------------------------------------------------------------------------

  protected array $extras = [];

  /**
   * ActivityDatesChangeEvent constructor.
   *
   * @param FileInterface $file
   * @param array $fileUsage
   */
  public function __construct(
      protected FileInterface $file,
      protected array $fileUsage
  ) {}

  /**
   * Set up any default extras values;
   *
   * @param array $extras
   *
   * @return array
   */
  protected function getDefaults(array $extras): array {
    $extras += [
      'now' => \Drupal::time()->getCurrentTime(),
    ];

    return $extras;
  }

  //---------------------------------------------------------------------------

  /**
   * @return FileInterface
   */
  public function getFile(): FileInterface {
    return $this->file;
  }

  /**
   * @return array
   */
  public function getFileUsage(): array {
    return $this->fileUsage;
  }

  /**
   * @return int
   */
  public function getNow(): int {
    return $this->get('now');
  }

  /**
   * @return array
   */
  public function getExtras(): array {
    return $this->extras;
  }

  /**
   * Get an event value.
   *
   * @param string $key
   * @param mixed $default
   *
   * @return mixed
   */
  public function get(string $key, mixed $default = NULL): mixed {
    return $this->extras[$key] ?? $default;
  }

  /**
   * Set an event value.
   *
   * @param string $key
   * @param mixed $value
   */
  public function set(string $key, mixed $value): void {
    $this->extras[$key] = $value;
  }

  //---------------------------------------------------------------------------

  /**
   * @return LoggerInterface
   */
  protected static function slogger(): LoggerInterface {
    return \Drupal::logger('file_event: ' . static::class);
  }

  /**
   * @return LoggerInterface
   */
  protected function logger(): LoggerInterface {
    return $this->getLogger('file_event: ' . get_class($this));
  }

}
