<?php

declare(strict_types=1);

namespace Drupal\untranslated_content_message\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;

/**
 * Provides an untranslated_content_message block.
 */
#[Block(
  id: "untranslated_content_message",
  admin_label: new TranslatableMarkup("Untranslated content message"),
  category: new TranslatableMarkup("System"),
)]
final class UntranslatedContentMessageBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the plugin instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Path\PathMatcherInterface $pathMatcher
   *   The path matcher.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $routeMatch
   *   The route match.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    private readonly LanguageManagerInterface $languageManager,
    private readonly PathMatcherInterface $pathMatcher,
    private readonly CurrentRouteMatch $routeMatch,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager'),
      $container->get('path.matcher'),
      $container->get('current_route_match'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [];
    if ($this->pathMatcher->isFrontPage() || !$this->routeMatch->getRouteObject()) {
      $url = Url::fromRoute('<front>');
    }
    else {
      $url = Url::fromRouteMatch($this->routeMatch);
    }
    // Use language content negotiation type or language interface.
    $defined_language_types = $this->languageManager->getLanguageTypes();
    $language_type = in_array('language_content', $defined_language_types) ? 'language_content' : 'language_interface';
    $links = $this->languageManager->getLanguageSwitchLinks($language_type, $url);
    $language_content_links = $links->links;

    // In any render cache items wrapping this block; account by user access
    // to each link, current url, path, query arguments, and language
    // negotiation.
    $cache_metadata = BubbleableMetadata::createFromRenderArray($build)
      ->addCacheContexts(['url.path', 'url.query_args', 'url.site', 'languages:' . $language_type]);
    if (isset($language_content_links)) {
      // Discover the entity we are currently viewing.
      // note:  page manager (and other) entities may need routines.
      foreach ($this->routeMatch->getParameters() as $param) {
        if ($param instanceof ContentEntityInterface && $param->isTranslatable()) {
          $entity = $param;
        }
      }

      if (!empty($entity)) {
        $no_translation_message = FALSE;
        foreach ($language_content_links as $lid => $link) {
          // Filter links in translated content only to show links to
          // existing translations.
          // Do not output anything if not a translatable entity.
          $has_translation = (method_exists($entity, 'getTranslationStatus')) ? $entity->getTranslationStatus($lid) : FALSE;
          $this_translation = ($has_translation && method_exists($entity, 'getTranslation')) ? $entity->getTranslation($lid) : FALSE;
          $access_translation = ($this_translation && method_exists($this_translation, 'access') && $this_translation->access('view')) ? TRUE : FALSE;
          if (!$this_translation || !$access_translation) {
            // If no translation for current language show the message.
            if ($this->languageManager->getCurrentLanguage($language_type)->getId() == $lid) {
              $no_translation_message = TRUE;
            }
            unset($language_content_links[$lid]);
          }

          if ($link['url'] instanceof Url) {
            $cache_metadata->addCacheableDependency($link['url']->access(NULL, TRUE));
          }
        }

        if ($no_translation_message) {
          $list = [
            '#theme' => 'links__untranslated_content_message_block',
            '#links' => $language_content_links,
            '#heading' => [
              'text' => $this->t('This content is not available in your selected language, so it is shown in its original language. The content is available in these languages:'),
              'level' => 'div',
              'attributes' => [
                'class' => [
                  'no-translation-message',
                ],
              ],
            ],
            '#attributes' => [
              'class' => [
                "language-switcher-untranslated-content",
              ],
            ],
            '#set_active_class' => TRUE,
          ];

          $build = [
            'message' => [
              '#theme' => 'status_messages',
              '#message_list' => [
                'warning' => [
                  $list,
                ],
              ],
              '#status_headings' => [
                'warning' => $this->t('Unavailable translation'),
              ],
            ],
          ];
        }
      }
    }
    $cache_metadata->applyTo($build);

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account): AccessResult {
    $access = $this->languageManager->isMultilingual() ? AccessResult::allowed() : AccessResult::forbidden();
    return $access->addCacheTags(['config:configurable_language_list']);
  }

  /**
   * {@inheritdoc}
   */
  public function createPlaceholder(): bool {
    return TRUE;
  }

}
