<?php

namespace Drupal\Tests\untranslated_content_message\Functional;

use Drupal\language\Entity\ConfigurableLanguage;

/**
 * Tests for the untranslated_content_message block.
 *
 * @group untranslated_content_message
 */
class ContentLanguageTest extends UntranslatedContentMessageTestBase {

  /**
   * The message warning of no translation available.
   *
   * @var string
   */
  protected $noTranslationMessage = 'This content is not available in your selected language, so it is shown in its original language. The content is available in these languages:';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Make both language_interface and language_content configurable:
    $this->config('language.types')->set('configurable', [
      'language_interface',
      'language_content',
    ])->save();
    // Enable URL language detection:
    $edit = [
      'language_content[enabled][language-url]' => TRUE,
      'language_content[weight][language-url]' => -10,
    ];
    $this->drupalGet('admin/config/regional/language/detection');
    $this->submitForm($edit, 'Save settings');
    $this->drupalGet('admin/config/regional/language/detection');
    // Create the untranslated content block:
    $this->placeBlock('untranslated_content_message', [
      'region' => 'content',
      'id' => 'test_language_content_block',
      'label_display' => FALSE,
    ]);
    // Create the language switcher block:
    $this->placeBlock('language_block:language_content', [
      'region' => 'content',
      'id' => 'language_switcher_content_block',
      'label_display' => FALSE,
    ]);
    // Enable translation for article bundle.
    \Drupal::service('content_translation.manager')->setEnabled('node', 'article', TRUE);
    $this->drupalGet('/admin/structure/block');
  }

  /**
   * See if the interface block does not exist on the front page.
   */
  public function testBlockNotExistsOnFront() {
    $session = $this->assertSession();

    $this->drupalGet('<front>');
    $session->statusCodeEquals(200);
    $session->pageTextContains('Test page text.');
    // Check if the block exists and contains the correct properties:
    $session->elementExists('css', '#block-test-language-content-block');
    // But as it is in front doesn't contain anything.
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block ul.language-switcher-untranslated-content');
  }

  /**
   * Check if the block does not exist on a non-node page.
   */
  public function testBlockNotExistsOnSystemPage() {
    $session = $this->assertSession();

    $this->drupalGet('/user/login');
    $session->statusCodeEquals(200);
    // Check if the block exists and contains the correct properties:
    $session->elementExists('css', '#block-test-language-content-block');
    // But as it is not content doesn't contain anything.
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block ul.language-switcher-untranslated-content');
  }

  /**
   * Check if the block doesn't exist on a not existing page.
   */
  public function testBlockNotExistsOnNonexistent() {
    $session = $this->assertSession();

    $this->drupalGet('/nonexistent page');
    $session->statusCodeEquals(404);
    // Check if the block exists:
    $session->elementExists('css', '#block-test-language-content-block');
    // But as it is not content doesn't contain anything.
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block ul.language-switcher-untranslated-content');
  }

  /**
   * Check if the block doesn't exist.
   *
   * Check if the block doesn't exist for anonymous user on a not accessible
   * admin page.
   */
  public function testBlockNotExistOnAdminAsAnon() {
    $session = $this->assertSession();

    $this->drupalLogout();
    $this->drupalGet('/admin');
    $session->statusCodeEquals(403);
    // Check if the block exists:
    $session->elementExists('css', '#block-test-language-content-block');
    // But as it is not content doesn't contain anything.
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block ul.language-switcher-untranslated-content');
  }

  /**
   * Test block works.
   *
   * Tests if the block exists but is empty on nodes when they are translated.
   */
  public function testBlockIsEmptyWhenTranslated() {
    $session = $this->assertSession();
    // Create node and translation:
    $node = $this->createNode([
      'type' => 'article',
      'id' => 1,
      'body' => 'English body content',
      'title' => 'English Version',
    ]);
    $node->addTranslation('de', [
      'body' => 'German body content',
      'title' => 'German Version',
    ])->save();
    // Check if the block exists but empty on a translated node:
    $this->drupalGet('/de/node/1');
    $session->pageTextContains('German body content');
    $session->elementExists('css', '#block-test-language-content-block');
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block a.language-link');

    // Check if the block exists but empty on the original node:
    $this->drupalGet('/node/1');
    $session->pageTextContains('English body content');
    $session->elementExists('css', '#block-test-language-content-block');
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block a.language-link');
  }

  /**
   * Tests if the block appears but empty on a not translatable node.
   */
  public function testBlockExistsButEmptyOnNonTranslatableContent() {
    $session = $this->assertSession();
    // Set article not translatable.
    \Drupal::service('content_translation.manager')->setEnabled('node', 'article', FALSE);
    // Create a node, but don't create a translation:
    $this->createNode([
      'type' => 'article',
      'id' => 1,
      'body' => 'Body content',
      'title' => 'English Version',
    ]);

    // Change the default language:
    $this->drupalGet('admin/config/regional/language');

    // Change the default language to german:
    $edit = [
      'site_default_language' => 'de',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->rebuildContainer();

    // Check if the block exists on the german version, which is not translated:
    $this->drupalGet('/node/1');
    $session->pageTextContains('Body content');
    // Bundle is not translatable, block must not exist.
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block a.language-link');
  }

  /**
   * Tests if the block exists but message is empty on original node.
   *
   * Tests if the block and message appears on a not translated
   * translatable node.
   */
  public function testBlockExistsOnTranslatableButNotTranslatedContent() {
    $session = $this->assertSession();
    // Create a node, but don't create a translation:
    $this->createNode([
      'type' => 'article',
      'id' => 1,
      'body' => 'Body content',
      'title' => 'English Version',
    ]);

    $this->drupalGet('/node/1');
    // Block exists but empty message on original content.
    $session->elementExists('css', '#block-test-language-content-block');
    $session->elementNotExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementNotExists('css', '#block-test-language-content-block a.language-link');

    // Change the default language:
    $this->drupalGet('admin/config/regional/language');

    // Change the default language to german:
    $edit = [
      'site_default_language' => 'de',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->rebuildContainer();

    // Check if the block exists on the german version, which is
    // not translated:
    $this->drupalGet('/node/1');
    $session->pageTextContains('Body content');
    // Bundle is translatable, block must exist as is available
    // only in english and we are in de:
    $session->elementTextEquals('css', '#block-test-language-content-block div.no-translation-message', $this->noTranslationMessage);
    $session->elementExists('css', '#block-test-language-content-block div.no-translation-message');
    $session->elementExists('css', '#block-test-language-content-block a.language-link');
  }

  /**
   * List available languages are visible for this content.
   */
  public function testAvailableLanguagesForThisContent() {
    $session = $this->assertSession();

    // Create a node, but don't create a translation:
    $node = $this->createNode([
      'type' => 'article',
      'id' => 1,
      'body' => 'English body content',
      'title' => 'English Version',
    ]);

    $node->addTranslation('de', [
      'body' => 'German body content',
      'title' => 'German Version',
    ])->save();

    // Create two more languages:
    ConfigurableLanguage::createFromLangcode('eo')->save();
    ConfigurableLanguage::createFromLangcode('fr')->save();

    $this->drupalGet('/node/1');
    $session->statusCodeEquals(200);
    $session->pageTextContains('English body content');
    // Block exists but empty message on original content.
    $session->elementExists('css', '#block-test-language-content-block');
    $session->elementNotExists('css', '#block-test-language-content-block a.language-link');

    $this->drupalGet('/de/node/1');
    $session->statusCodeEquals(200);
    $session->pageTextContains('German body content');
    // Block exists but empty message on original content.
    $session->elementExists('css', '#block-test-language-content-block');
    $session->elementNotExists('css', '#block-test-language-content-block a.language-link');

    $this->drupalGet('/eo/node/1');
    $session->statusCodeEquals(200);
    $session->pageTextContains('English body content');

    // Check if the block exists and contains the correct properties:
    $session->elementExists('css', '#block-test-language-content-block');
    $session->elementTextEquals('css', '#block-test-language-content-block ul.language-switcher-untranslated-content > li:nth-child(1) a.language-link', 'English');
    $session->elementTextEquals('css', '#block-test-language-content-block ul.language-switcher-untranslated-content > li:nth-child(2) a.language-link', 'German');
    // No more than these 2 languages must exist in links.
    $session->elementNotExists('css', '#block-test-language-content-block ul.language-switcher-untranslated-content > li:nth-child(3) a.language-link');
  }

}
