<?php

namespace Drupal\upsc_quiz\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure UPSC Quiz settings.
 */
class AdminSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'upsc_quiz_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'upsc_quiz.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('upsc_quiz.settings');

    $form['general'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('General Settings'),
    ];

    $form['general']['time_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Time Limit (minutes)'),
      '#description' => $this->t('Default time limit for exam mode in minutes.'),
      '#default_value' => $config->get('time_limit') ?: 90,
      '#min' => 1,
      '#max' => 300,
      '#required' => TRUE,
    ];

    $form['general']['enable_analytics'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Analytics'),
      '#description' => $this->t('Track user performance and generate analytics.'),
      '#default_value' => $config->get('enable_analytics') !== FALSE,
    ];

    $form['general']['allow_retake'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow Retake'),
      '#description' => $this->t('Allow users to retake quizzes multiple times.'),
      '#default_value' => $config->get('allow_retake') !== FALSE,
    ];

    $form['general']['show_correct_answers'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Correct Answers'),
      '#description' => $this->t('Show correct answers and explanations after quiz completion.'),
      '#default_value' => $config->get('show_correct_answers') !== FALSE,
    ];

    $form['questions'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Question Settings'),
    ];

    $form['questions']['randomize_questions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Randomize Questions'),
      '#description' => $this->t('Randomize the order of questions in each quiz attempt.'),
      '#default_value' => $config->get('randomize_questions') ?: FALSE,
    ];

    $form['questions']['randomize_options'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Randomize Options'),
      '#description' => $this->t('Randomize the order of answer options for each question.'),
      '#default_value' => $config->get('randomize_options') ?: FALSE,
    ];

    $sections = [
      'reasoning' => $this->t('Reasoning & Logic'),
      'english' => $this->t('English Language'),
      'polity' => $this->t('Indian Polity'),
      'history' => $this->t('History'),
      'geography' => $this->t('Geography'),
      'current-affairs' => $this->t('Current Affairs'),
    ];

    $form['questions']['questions_per_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Questions per Section'),
      '#description' => $this->t('Set the number of questions to display for each section. Leave empty to show all questions.'),
    ];

    $questions_per_section = $config->get('questions_per_section') ?: [];
    foreach ($sections as $key => $label) {
      $form['questions']['questions_per_section'][$key] = [
        '#type' => 'number',
        '#title' => $label,
        '#default_value' => $questions_per_section[$key] ?? '',
        '#min' => 0,
        '#max' => 100,
      ];
    }

    $form['email'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Email Notifications'),
    ];

    $form['email']['notify_admin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Notify Admin'),
      '#description' => $this->t('Send email notifications to admin when users complete quizzes.'),
      '#default_value' => $config->get('notify_admin') ?: FALSE,
    ];

    $form['email']['admin_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Admin Email'),
      '#description' => $this->t('Email address to receive notifications.'),
      '#default_value' => $config->get('admin_email') ?: '',
      '#states' => [
        'visible' => [
          ':input[name="notify_admin"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="notify_admin"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['advanced'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Advanced Settings'),
    ];

    $form['advanced']['cache_questions'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cache Questions'),
      '#description' => $this->t('Cache questions in browser localStorage for faster loading.'),
      '#default_value' => $config->get('cache_questions') !== FALSE,
    ];

    $form['advanced']['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug Mode'),
      '#description' => $this->t('Enable debug mode for development purposes. Do not enable on production sites.'),
      '#default_value' => $config->get('debug_mode') ?: FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate that admin email is provided if notifications are enabled
    if ($form_state->getValue('notify_admin') && empty($form_state->getValue('admin_email'))) {
      $form_state->setErrorByName('admin_email', $this->t('Admin email is required when notifications are enabled.'));
    }

    // Validate questions per section values
    $questions_per_section = $form_state->getValue('questions_per_section');
    foreach ($questions_per_section as $section => $count) {
      if (!empty($count) && $count < 1) {
        $form_state->setErrorByName("questions_per_section][$section]", 
          $this->t('Questions per section must be at least 1 if specified.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('upsc_quiz.settings');
    
    $config
      ->set('time_limit', $form_state->getValue('time_limit'))
      ->set('enable_analytics', $form_state->getValue('enable_analytics'))
      ->set('allow_retake', $form_state->getValue('allow_retake'))
      ->set('show_correct_answers', $form_state->getValue('show_correct_answers'))
      ->set('randomize_questions', $form_state->getValue('randomize_questions'))
      ->set('randomize_options', $form_state->getValue('randomize_options'))
      ->set('questions_per_section', array_filter($form_state->getValue('questions_per_section') ?: []))
      ->set('notify_admin', $form_state->getValue('notify_admin'))
      ->set('admin_email', $form_state->getValue('admin_email'))
      ->set('cache_questions', $form_state->getValue('cache_questions'))
      ->set('debug_mode', $form_state->getValue('debug_mode'))
      ->save();

    parent::submitForm($form, $form_state);
    
    // Clear relevant caches
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['upsc_quiz:settings']);
  }

}