<?php

namespace Drupal\urct;

use Drupal\Core\PathProcessor\InboundPathProcessorInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\user_referral\Entity\UserReferralType;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;

class ReferralUrlHandler implements InboundPathProcessorInterface {

  // const COOKIE_NAME = 'urct_referral';

  /**
   * Referrer user id and referral type.
   *
   * @var \Drupal\urct\ReferralItem
   */
  protected $referralItem;

  protected $processed = FALSE;

  public static $setting_path_cookie = FALSE;

  /**
   * The stream wrapper manager service.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * Constructs a new PathProcessorImageStyles object.
   *
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager service.
   */
  public function __construct(StreamWrapperManagerInterface $stream_wrapper_manager) {
    $this->streamWrapperManager = $stream_wrapper_manager;
  }

  /**
   * @return \Drupal\urct\ReferralItem
   */
  public static function getReferralFromPath($path) {
    // We parse the path to avoid any issue when path is an absolute URL.
    $parsed_url = parse_url($path);
    $path = $parsed_url['path'];

    $path_parts = array_filter(explode('/', $path));
    $path_part_count = count($path_parts);
    $original_path_parts = $path_parts;
    $num_items_to_remove = 0;
    if ($path_part_count > 0) {
      $account = NULL;
      $referral_type = NULL;
      $refid = NULL;
      $last_item = array_pop($path_parts);
      if ( $path_part_count > 1) {
        // Path requires 2 or more components to have referral type name in its last part.
        // Last part may be referrer id or referral type.
        // Let's check first if it is a referral type.
        $referral_type = UserReferralType::load($last_item);
        if ($referral_type) {
          $previous_to_last_item = array_pop($path_parts);
          $account = $referral_type->getReferralIDAccount($previous_to_last_item);
          if ($account) {
            $refid = $previous_to_last_item;
            $num_items_to_remove = 2;
            if ($path_part_count > 2 && end($path_parts) == 'direct') {
              $num_items_to_remove = 3;
            }
          }
        }
      }
      if (!$account) {
        // Last path part was not referral type name. It might be referrer ID itelf.
        // Get 'default' referral type by order.
        $referral_types = UserReferralType::loadMultiple();
        $referral_type = reset($referral_types);
        if ($referral_type) {
          $account = $referral_type->getReferralIDAccount($last_item);
          $refid = $last_item;
          $num_items_to_remove = 1;
          if ($path_part_count > 1 && end($path_parts) == 'direct') {
            $num_items_to_remove = 2;
          }
        }
      }
      if ($account && $referral_type) {
        $referral_item = new ReferralItem();
        $referral_item->referral_id = $refid;
        $referral_item->uid = $account->id();
        $referral_item->type = $referral_type->id();
        $referral_item->normal_path = '/' . implode('/', array_slice($original_path_parts, 0, $path_part_count - $num_items_to_remove));
        $referral_item->referral_id_only = $num_items_to_remove == 1;
        if (UserReferralType::count() == 1) {
          // Force to use referrer id only in the path if there is only one referral type.
          $referral_item->referral_id_only = TRUE;
        }

        return $referral_item;
      }
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function processInbound($path, Request $request) {
    /** @var \Drupal\Core\StreamWrapper\PublicStream */
    $stream_wrapper = $this->streamWrapperManager->getViaScheme('public');
    $directory_path = $stream_wrapper->getDirectoryPath();
    if (strpos($path, '/' . $directory_path . '/styles/') === 0) {
      // This is image style path.
      // Thus remove Referrel from end of the file set in request by
      // Drupal\image\PathProcessor\PathProcessorImageStyles::processInbound()
      $file = $request->query->get('file');
      $result = static::getReferralFromPath($file);
      if ($result) {
        $request->query->set('file', ltrim($result->normal_path, '/'));
      }
    }

    if (\Drupal::service('urct.path_validator')->getUrlIfValidWithoutAccessCheck($path)) {
      return $path;
    }
    // if (strpos($path, '/user/') === 0) {
    //   return $path;
    // }
    $result = static::getReferralFromPath($path);
    if ($result) {
      $path = $result->normal_path;
      $this->referralItem = $result;

      $request->attributes->add(['_disable_route_normalizer' => TRUE]);
      $this->setPathReferralCookie($this->referralItem, TRUE);
      \Drupal::service('urct.referral_manager')->setCurrentReferralItem($result);

      $this->processed = TRUE;
    }
    return $path;
  }

  public static function setPathReferralCookie(ReferralItem $referral_item, $overwrite = FALSE) {
    static $set_cookie = FALSE;
    if (!$set_cookie) {
      $referral_type = UserReferralType::load($referral_item->type);
      if ($referral_type) {
        $account = $referral_type->getReferralIDAccount($referral_item->referral_id);
        if ($account) {
          $request = \Drupal::request();

          $cookie_data = $request->cookies->get(UserReferralType::COOKIE_NAME);
          $existing_cookie = isset($cookie_data) ? json_decode($cookie_data) : NULL;
          if ($existing_cookie) {
            if (empty($existing_cookie->auto)) {
              // Not to process cookie set with regular referral link or aliases.
              return;
            }
            else if ($overwrite) {
              // This cookie is with auto assigned referrer which is supposed to overwirte.
              // So remove it from $_COOKIE array, other wise UserReferralType::setCookie() will apply reassign logic.
              $request->cookies->remove(UserReferralType::COOKIE_NAME);
            }
          }
          if ($overwrite || !$existing_cookie) {
            self::$setting_path_cookie = TRUE;
            // $cookie = new \stdClass();
            // $cookie->uid = $account->id();
            // $cookie->type = $referral_item->type;
            // setcookie(self::COOKIE_NAME, json_encode($cookie), time() + 7 * 24 * 60 * 60, '/');
            $referral_type->setCookie($account, \Drupal::request());
            $set_cookie = TRUE;
            self::$setting_path_cookie = FALSE;
          }
        }
      }
    }
  }

  public function isProcessed() {
    return $this->processed;
  }

  public function setProcessed($processed) {
    return $this->processed = $processed;
  }
}