<?php

namespace Drupal\url_path_restrictions\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Routing\RouteBuildEvent;
use Drupal\Core\Routing\RoutingEvents;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Validates routes against disallowed URL patterns.
 */
class RouteValidationSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger factory service.
   */
  protected LoggerChannelFactoryInterface $loggerFactory;

  /**
   * Constructs a RouteValidationSubscriber object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory) {
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      RoutingEvents::ALTER => ['onRouteAlter', 0],
    ];
  }

  /**
   * Validates routes during route collection alteration.
   */
  public function onRouteAlter(RouteBuildEvent $event): void {
    $collection = $event->getRouteCollection();
    $config = $this->configFactory->get('url_path_restrictions.settings');
    $disallowed_patterns = $config->get('disallowed_patterns') ?: [];

    if (empty($disallowed_patterns)) {
      return;
    }

    foreach ($collection->all() as $route_name => $route) {
      $path = $route->getPath();

      foreach ($disallowed_patterns as $pattern) {
        if ($this->matchesPattern($path, $pattern)) {
          // Log the violation.
          $this->loggerFactory->get('url_path_restrictions')->error(
            'Route "@route_name" with path "@path" matches disallowed pattern "@pattern". Route removed.',
            [
              '@route_name' => $route_name,
              '@path' => $path, 
              '@pattern' => $pattern,
            ]
          );

          // Remove the route from the collection to prevent it from being used.
          $collection->remove($route_name);
          break;
        }
      }
    }
  }

  /**
   * Checks if a path is disallowed.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if the path is disallowed, FALSE otherwise.
   */
  protected function isDisallowedPath(string $path): bool {
    $config = $this->configFactory->get('url_path_restrictions.settings');
    $disallowed_patterns = $config->get('disallowed_patterns') ?: [];

    foreach ($disallowed_patterns as $pattern) {
      if ($this->matchesPattern($path, $pattern)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Checks if a URL matches a pattern.
   *
   * @param string $url
   *   The URL to check.
   * @param string $pattern
   *   The pattern to match against.
   *
   * @return bool
   *   TRUE if the URL matches the pattern, FALSE otherwise.
   */
  protected function matchesPattern(string $url, string $pattern): bool {
    // Normalize URLs by ensuring they start with a slash.
    $url = '/' . ltrim($url, '/');
    $pattern = '/' . ltrim($pattern, '/');

    // Convert wildcard pattern to regex.
    // Escape special regex characters except *.
    $regex_pattern = preg_quote($pattern, '/');
    // Replace \* with .* for wildcard matching.
    $regex_pattern = str_replace('\*', '.*', $regex_pattern);
    // Anchor the pattern.
    $regex_pattern = '/^' . $regex_pattern . '$/';

    return (bool) preg_match($regex_pattern, $url);
  }

}