<?php

namespace Drupal\url_path_restrictions\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for URL path restrictions settings.
 */
class UrlPathRestrictionsSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['url_path_restrictions.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'url_path_restrictions_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('url_path_restrictions.settings');

    $form['disallowed_patterns'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Disallowed URL patterns'),
      '#description' => $this->t('Enter one URL pattern per line. All patterns must start with a forward slash (/). Use asterisks (*) as wildcards. Examples:<br>/projects<br>/projects/*<br>/api<br>/*/reserved/*'),
      '#default_value' => implode("\n", $config->get('disallowed_patterns') ?: []),
      '#rows' => 10,
    ];

    $form['help'] = [
      '#type' => 'details',
      '#title' => $this->t('Pattern examples'),
      '#description' => $this->t('Here are some examples of how patterns work:'),
    ];

    $form['help']['examples'] = [
      '#theme' => 'item_list',
      '#title' => $this->t('Examples:'),
      '#items' => [
        $this->t('<code>/projects</code> - Matches exactly "/projects"'),
        $this->t('<code>/projects/*</code> - Matches "/projects" and any path starting with "/projects/"'),
        $this->t('<code>/api</code> - Matches exactly "/api"'),
        $this->t('<code>/*/reserved/*</code> - Matches any path containing "/reserved/" anywhere in the path'),
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $patterns = $form_state->getValue('disallowed_patterns');
    if (!empty($patterns)) {
      $lines = explode("\n", $patterns);
      foreach ($lines as $line_number => $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }

        // Ensure all patterns start with a forward slash.
        if (!str_starts_with($line, '/')) {
          $form_state->setErrorByName('disallowed_patterns', 
            $this->t('Line @line: "@pattern" must start with a forward slash (/).', [
              '@line' => $line_number + 1,
              '@pattern' => $line,
            ])
          );
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $patterns = $form_state->getValue('disallowed_patterns');
    
    // Process the textarea input into an array.
    $patterns_array = [];
    if (!empty($patterns)) {
      $lines = explode("\n", $patterns);
      foreach ($lines as $line) {
        $line = trim($line);
        if (!empty($line)) {
          $patterns_array[] = $line;
        }
      }
    }

    $this->config('url_path_restrictions.settings')
      ->set('disallowed_patterns', $patterns_array)
      ->save();

    parent::submitForm($form, $form_state);
  }

}