<?php

namespace Drupal\url_path_restrictions\Plugin\Validation\Constraint;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates the DisallowedUrlPattern constraint.
 */
class DisallowedUrlPatternConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The config factory service.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new DisallowedUrlPatternConstraintValidator object.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate(mixed $value, Constraint $constraint): void {
    if (!$constraint instanceof DisallowedUrlPatternConstraint) {
      return;
    }

    if (empty($value)) {
      return;
    }

    // Extract the string value from the field item.
    $alias_value = '';
    if (is_string($value)) {
      $alias_value = $value;
    }
    elseif ($value instanceof FieldItemListInterface) {
      // Handle FieldItemList - get the first item's value.
      $items = $value->getValue();
      if (!empty($items) && isset($items[0]['value'])) {
        $alias_value = $items[0]['value'];
      }
    }

    if (empty($alias_value)) {
      return;
    }

    $config = $this->configFactory->get('url_path_restrictions.settings');
    $disallowed_patterns = $config->get('disallowed_patterns') ?: [];

    foreach ($disallowed_patterns as $pattern) {
      if ($this->matchesPattern($alias_value, $pattern)) {
        $this->context->addViolation($constraint->message, [
          '@alias' => $alias_value,
          '@pattern' => $pattern,
        ]);
        break;
      }
    }
  }

  /**
   * Checks if a URL matches a pattern.
   *
   * @param string $url
   *   The URL to check.
   * @param string $pattern
   *   The pattern to match against.
   *
   * @return bool
   *   TRUE if the URL matches the pattern, FALSE otherwise.
   */
  protected function matchesPattern(string $url, string $pattern): bool {
    // Normalize URLs by ensuring they start with a slash.
    $url = '/' . ltrim($url, '/');
    $pattern = '/' . ltrim($pattern, '/');

    // Convert wildcard pattern to regex.
    // Escape special regex characters except *.
    $regex_pattern = preg_quote($pattern, '/');
    // Replace \* with .* for wildcard matching.
    $regex_pattern = str_replace('\*', '.*', $regex_pattern);
    // Anchor the pattern.
    $regex_pattern = '/^' . $regex_pattern . '$/';

    return (bool) preg_match($regex_pattern, $url);
  }

}