<?php

namespace Drupal\Tests\url_path_restrictions\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\path_alias\Entity\PathAlias;

/**
 * Tests URL alias validation integration.
 *
 * @group url_path_restrictions
 */
class UrlAliasValidationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'url_path_restrictions',
    'path_alias',
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('path_alias');
    $this->installConfig(['url_path_restrictions']);
  }

  /**
   * Tests that disallowed URL aliases fail validation.
   */
  public function testDisallowedUrlAliasValidation(): void {
    // Configure disallowed patterns.
    $config = $this->config('url_path_restrictions.settings');
    $config->set('disallowed_patterns', ['/projects', '/projects/*', '/api', '/*/reserved/*']);
    $config->save();

    // Test exact match.
    $path_alias = PathAlias::create([
      'path' => '/node/1',
      'alias' => '/projects',
    ]);

    $violations = $path_alias->validate();
    $this->assertCount(1, $violations);
    $this->assertStringContainsString('matches a disallowed pattern', (string) $violations[0]->getMessage());

    // Test wildcard match.
    $path_alias = PathAlias::create([
      'path' => '/node/2',
      'alias' => '/projects/mine',
    ]);

    $violations = $path_alias->validate();
    $this->assertCount(1, $violations);
    $this->assertStringContainsString('/projects/*', (string) $violations[0]->getMessage());

    // Test infix wildcard match.
    $path_alias = PathAlias::create([
      'path' => '/node/3',
      'alias' => '/my-pages/reserved/cats.html',
    ]);

    $violations = $path_alias->validate();
    $this->assertCount(1, $violations);
    $this->assertStringContainsString('/*/reserved/*', (string) $violations[0]->getMessage());
  }

  /**
   * Tests that allowed URL aliases pass validation.
   */
  public function testAllowedUrlAliasValidation(): void {
    // Configure disallowed patterns.
    $config = $this->config('url_path_restrictions.settings');
    $config->set('disallowed_patterns', ['/projects', '/projects/*', '/api']);
    $config->save();

    // Test allowed alias.
    $path_alias = PathAlias::create([
      'path' => '/node/1',
      'alias' => '/allowed/path',
    ]);

    $violations = $path_alias->validate();
    $this->assertCount(0, $violations);

    // Test that partial matches don't trigger validation.
    $path_alias = PathAlias::create([
      'path' => '/node/2',
      'alias' => '/project-info',
    ]);

    $violations = $path_alias->validate();
    $this->assertCount(0, $violations);
  }

  /**
   * Tests validation with no disallowed patterns configured.
   */
  public function testValidationWithNoPatterns(): void {
    // Ensure no patterns are configured.
    $config = $this->config('url_path_restrictions.settings');
    $config->set('disallowed_patterns', []);
    $config->save();

    $path_alias = PathAlias::create([
      'path' => '/node/1',
      'alias' => '/projects',
    ]);

    $violations = $path_alias->validate();
    $this->assertCount(0, $violations);
  }

  /**
   * Tests that the constraint is properly applied to the alias field.
   */
  public function testConstraintIsAppliedToAliasField(): void {
    $path_alias = PathAlias::create([
      'path' => '/node/1',
      'alias' => '/test',
    ]);

    // Get the field definition and check that our constraint is applied.
    $field_definition = $path_alias->getFieldDefinition('alias');
    $constraints = $field_definition->getConstraints();
    
    $this->assertArrayHasKey('DisallowedUrlPattern', $constraints);
  }

}