<?php

namespace Drupal\Tests\url_path_restrictions\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\url_path_restrictions\Plugin\Validation\Constraint\DisallowedUrlPatternConstraint;
use Drupal\url_path_restrictions\Plugin\Validation\Constraint\DisallowedUrlPatternConstraintValidator;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Tests the DisallowedUrlPatternConstraintValidator.
 *
 * @coversDefaultClass \Drupal\url_path_restrictions\Plugin\Validation\Constraint\DisallowedUrlPatternConstraintValidator
 * @group url_path_restrictions
 */
class DisallowedUrlPatternConstraintValidatorTest extends UnitTestCase {

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The validator under test.
   *
   * @var \Drupal\url_path_restrictions\Plugin\Validation\Constraint\DisallowedUrlPatternConstraintValidator
   */
  protected $validator;

  /**
   * The execution context mock.
   *
   * @var \Symfony\Component\Validator\Context\ExecutionContextInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $context;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->validator = new DisallowedUrlPatternConstraintValidator($this->configFactory);
    $this->context = $this->createMock(ExecutionContextInterface::class);
    $this->validator->initialize($this->context);
  }

  /**
   * Tests validation with no disallowed patterns.
   *
   * @covers ::validate
   */
  public function testValidateWithNoPatterns(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn([]);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $constraint = new DisallowedUrlPatternConstraint();

    $this->context->expects($this->never())
      ->method('addViolation');

    $this->validator->validate('/some/path', $constraint);
  }

  /**
   * Tests validation with empty value.
   *
   * @covers ::validate
   */
  public function testValidateWithEmptyValue(): void {
    $constraint = new DisallowedUrlPatternConstraint();

    $this->configFactory->expects($this->never())
      ->method('get');

    $this->context->expects($this->never())
      ->method('addViolation');

    $this->validator->validate('', $constraint);
  }

  /**
   * Tests validation with string value that matches pattern.
   *
   * @covers ::validate
   * @covers ::matchesPattern
   */
  public function testValidateStringValueMatchesPattern(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn(['/projects', '/api']);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $constraint = new DisallowedUrlPatternConstraint();

    $this->context->expects($this->once())
      ->method('addViolation')
      ->with($constraint->message, [
        '@alias' => '/projects',
        '@pattern' => '/projects',
      ]);

    $this->validator->validate('/projects', $constraint);
  }

  /**
   * Tests validation with FieldItemList value.
   *
   * @covers ::validate
   */
  public function testValidateFieldItemListValue(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn(['/projects/*']);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $fieldItemList = $this->createMock(FieldItemListInterface::class);
    $fieldItemList->expects($this->once())
      ->method('getValue')
      ->willReturn([['value' => '/projects/mine']]);

    $constraint = new DisallowedUrlPatternConstraint();

    $this->context->expects($this->once())
      ->method('addViolation')
      ->with($constraint->message, [
        '@alias' => '/projects/mine',
        '@pattern' => '/projects/*',
      ]);

    $this->validator->validate($fieldItemList, $constraint);
  }

  /**
   * Tests validation with string value that doesn't match pattern.
   *
   * @covers ::validate
   * @covers ::matchesPattern
   */
  public function testValidateStringValueDoesNotMatch(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn(['/projects', '/api']);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $constraint = new DisallowedUrlPatternConstraint();

    $this->context->expects($this->never())
      ->method('addViolation');

    $this->validator->validate('/allowed/path', $constraint);
  }

  /**
   * Tests pattern matching with wildcards.
   *
   * @covers ::matchesPattern
   * @dataProvider wildcardPatternProvider
   */
  public function testWildcardPatternMatching(string $pattern, string $url, bool $shouldMatch): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn([$pattern]);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $constraint = new DisallowedUrlPatternConstraint();

    if ($shouldMatch) {
      $this->context->expects($this->once())
        ->method('addViolation');
    } else {
      $this->context->expects($this->never())
        ->method('addViolation');
    }

    $this->validator->validate($url, $constraint);
  }

  /**
   * Data provider for wildcard pattern testing.
   *
   * @return array
   *   Test cases with pattern, URL, and expected match result.
   */
  public static function wildcardPatternProvider(): array {
    return [
      'exact match' => ['/projects', '/projects', TRUE],
      'wildcard suffix match' => ['/projects/*', '/projects/mine', TRUE],
      'wildcard suffix no match' => ['/projects/*', '/other/path', FALSE],
      'wildcard infix match' => ['/*/reserved/*', '/my-pages/reserved/cats.html', TRUE],
      'wildcard infix no match' => ['/*/reserved/*', '/my-pages/public/cats.html', FALSE],
      'multiple wildcards' => ['/*/*/test', '/path/sub/test', TRUE],
      'normalize leading slash' => ['/api', 'api', TRUE],
    ];
  }

}