<?php

namespace Drupal\Tests\url_path_restrictions\Unit;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Routing\RouteBuildEvent;
use Drupal\Core\Routing\RoutingEvents;
use Drupal\Tests\UnitTestCase;
use Drupal\url_path_restrictions\EventSubscriber\RouteValidationSubscriber;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Tests the RouteValidationSubscriber.
 *
 * @coversDefaultClass \Drupal\url_path_restrictions\EventSubscriber\RouteValidationSubscriber
 * @group url_path_restrictions
 */
class RouteValidationSubscriberTest extends UnitTestCase {

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The logger factory mock.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * The logger channel mock.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The subscriber under test.
   *
   * @var \Drupal\url_path_restrictions\EventSubscriber\RouteValidationSubscriber
   */
  protected $subscriber;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);

    $this->loggerFactory->expects($this->any())
      ->method('get')
      ->with('url_path_restrictions')
      ->willReturn($this->logger);

    $this->subscriber = new RouteValidationSubscriber($this->configFactory, $this->loggerFactory);
  }

  /**
   * Tests that the subscriber listens to the correct event.
   *
   * @covers ::getSubscribedEvents
   */
  public function testGetSubscribedEvents(): void {
    $events = RouteValidationSubscriber::getSubscribedEvents();
    $this->assertArrayHasKey(RoutingEvents::ALTER, $events);
    $this->assertEquals(['onRouteAlter', 0], $events[RoutingEvents::ALTER]);
  }

  /**
   * Tests route validation with no disallowed patterns.
   *
   * @covers ::onRouteAlter
   */
  public function testOnRouteAlterWithNoPatterns(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn([]);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $collection = new RouteCollection();
    $collection->add('test.route', new Route('/test/path'));

    $event = $this->createMock(RouteBuildEvent::class);
    $event->expects($this->once())
      ->method('getRouteCollection')
      ->willReturn($collection);

    $this->logger->expects($this->never())
      ->method('error');

    $this->subscriber->onRouteAlter($event);

    // Ensure the route is still in the collection.
    $this->assertTrue($collection->get('test.route') !== null);
  }

  /**
   * Tests route validation with matching disallowed pattern.
   *
   * @covers ::onRouteAlter
   * @covers ::matchesPattern
   */
  public function testOnRouteAlterWithMatchingPattern(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn(['/projects', '/api/*']);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $collection = new RouteCollection();
    $collection->add('projects.route', new Route('/projects'));
    $collection->add('allowed.route', new Route('/allowed/path'));

    $event = $this->createMock(RouteBuildEvent::class);
    $event->expects($this->once())
      ->method('getRouteCollection')
      ->willReturn($collection);

    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        'Route "@route_name" with path "@path" matches disallowed pattern "@pattern". Route removed.',
        [
          '@route_name' => 'projects.route',
          '@path' => '/projects',
          '@pattern' => '/projects',
        ]
      );

    $this->subscriber->onRouteAlter($event);

    // Ensure the disallowed route is removed.
    $this->assertNull($collection->get('projects.route'));
    // Ensure the allowed route remains.
    $this->assertNotNull($collection->get('allowed.route'));
  }

  /**
   * Tests route validation with wildcard patterns.
   *
   * @covers ::onRouteAlter
   * @covers ::matchesPattern
   */
  public function testOnRouteAlterWithWildcardPatterns(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn(['/api/*', '/*/admin/*']);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $collection = new RouteCollection();
    $collection->add('api.v1', new Route('/api/v1/users'));
    $collection->add('admin.route', new Route('/system/admin/config'));
    $collection->add('allowed.route', new Route('/public/info'));

    $event = $this->createMock(RouteBuildEvent::class);
    $event->expects($this->once())
      ->method('getRouteCollection')
      ->willReturn($collection);

    $this->logger->expects($this->exactly(2))
      ->method('error');

    $this->subscriber->onRouteAlter($event);

    // Ensure the disallowed routes are removed.
    $this->assertNull($collection->get('api.v1'));
    $this->assertNull($collection->get('admin.route'));
    // Ensure the allowed route remains.
    $this->assertNotNull($collection->get('allowed.route'));
  }

  /**
   * Tests that only the first matching pattern is logged per route.
   *
   * @covers ::onRouteAlter
   */
  public function testOnRouteAlterLogsOnlyFirstMatch(): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn(['/projects', '/projects/*']);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $collection = new RouteCollection();
    $collection->add('projects.route', new Route('/projects'));

    $event = $this->createMock(RouteBuildEvent::class);
    $event->expects($this->once())
      ->method('getRouteCollection')
      ->willReturn($collection);

    // Should only log once, for the first matching pattern.
    $this->logger->expects($this->once())
      ->method('error')
      ->with(
        'Route "@route_name" with path "@path" matches disallowed pattern "@pattern". Route removed.',
        [
          '@route_name' => 'projects.route',
          '@path' => '/projects',
          '@pattern' => '/projects',
        ]
      );

    $this->subscriber->onRouteAlter($event);
  }

  /**
   * Tests pattern matching logic.
   *
   * @covers ::matchesPattern
   * @dataProvider patternMatchingProvider
   */
  public function testPatternMatching(string $pattern, string $path, bool $shouldMatch): void {
    $config = $this->createMock(ImmutableConfig::class);
    $config->expects($this->once())
      ->method('get')
      ->with('disallowed_patterns')
      ->willReturn([$pattern]);

    $this->configFactory->expects($this->once())
      ->method('get')
      ->with('url_path_restrictions.settings')
      ->willReturn($config);

    $collection = new RouteCollection();
    $collection->add('test.route', new Route($path));

    $event = $this->createMock(RouteBuildEvent::class);
    $event->expects($this->once())
      ->method('getRouteCollection')
      ->willReturn($collection);

    if ($shouldMatch) {
      $this->logger->expects($this->once())
        ->method('error');
    } else {
      $this->logger->expects($this->never())
        ->method('error');
    }

    $this->subscriber->onRouteAlter($event);

    if ($shouldMatch) {
      $this->assertNull($collection->get('test.route'));
    } else {
      $this->assertNotNull($collection->get('test.route'));
    }
  }

  /**
   * Data provider for pattern matching tests.
   *
   * @return array
   *   Test cases with pattern, path, and expected match result.
   */
  public static function patternMatchingProvider(): array {
    return [
      'exact match' => ['/api', '/api', TRUE],
      'wildcard suffix match' => ['/api/*', '/api/v1/users', TRUE],
      'wildcard suffix no match' => ['/api/*', '/public/api', FALSE],
      'wildcard infix match' => ['/*/admin/*', '/system/admin/config', TRUE],
      'wildcard infix no match' => ['/*/admin/*', '/system/public/config', FALSE],
      'no match different path' => ['/projects', '/products', FALSE],
    ];
  }

}