/**
 * @file
 * Email template preview functionality for user account settings.
 *
 * Provides a tabbed interface (Template/Preview) for email templates on
 * /admin/config/people/accounts. When the Preview tab is clicked, the
 * email body content is sent to a preview controller and rendered in
 * an iframe with token replacement.
 *
 * @module user_email_preview
 */

(function (Drupal, drupalSettings, once) {
  'use strict';

  /**
   * CSS class constants.
   *
   * @constant {Object}
   */
  const CLASSES = {
    TABS_CONTAINER: 'email-template-tabs',
    TAB_LINK: 'tabs__link',
    TAB_CONTENT: 'email-tab-content',
    TAB_CONTENT_ACTIVE: 'email-tab-content--active',
    TAB_ACTIVE: 'is-active',
  };

  /**
   * Data attribute constants.
   *
   * @constant {Object}
   */
  const DATA_ATTRS = {
    BODY_FIELD: 'data-body-field',
    SUBJECT_FIELD: 'data-subject-field',
    IFRAME_ID: 'data-iframe-id',
    TAB: 'data-tab',
    CONTENT: 'data-content',
  };

  /**
   * Tab type constants.
   *
   * @constant {Object}
   */
  const TABS = {
    TEMPLATE: 'template',
    PREVIEW: 'preview',
  };

  /**
   * Creates an EmailTemplatePreview instance for a tabs container.
   *
   * @class
   * @param {HTMLElement} container - The tabs container element.
   * @param {string} previewUrl - The URL for the preview endpoint.
   */
  function EmailTemplatePreview(container, previewUrl) {
    this.container = container;
    this.previewUrl = previewUrl;
    this.bodyFieldName = container.getAttribute(DATA_ATTRS.BODY_FIELD);
    this.subjectFieldName = container.getAttribute(DATA_ATTRS.SUBJECT_FIELD);
    this.iframe = document.getElementById(
      container.getAttribute(DATA_ATTRS.IFRAME_ID)
    );
    this.tabs = container.querySelectorAll('.' + CLASSES.TAB_LINK);
    this.contents = container.querySelectorAll('.' + CLASSES.TAB_CONTENT);

    this.init();
  }

  /**
   * Initializes the email template preview.
   */
  EmailTemplatePreview.prototype.init = function () {
    if (!this.iframe) {
      return;
    }

    this.bindEvents();
  };

  /**
   * Binds event listeners.
   */
  EmailTemplatePreview.prototype.bindEvents = function () {
    const self = this;

    this.iframe.addEventListener('load', function () {
      self.adjustIframeHeight();
    });

    this.tabs.forEach(function (tab) {
      tab.addEventListener('click', function (event) {
        self.handleTabClick(event, tab);
      });
    });
  };

  /**
   * Adjusts the iframe height to match its content.
   *
   * Attempts to read the content height from the iframe document.
   * Falls back to CSS min-height if cross-origin restrictions apply.
   */
  EmailTemplatePreview.prototype.adjustIframeHeight = function () {
    try {
      const iframeDoc =
        this.iframe.contentDocument || this.iframe.contentWindow.document;

      if (iframeDoc && iframeDoc.body) {
        const height = iframeDoc.body.scrollHeight;
        this.iframe.style.height = height + 'px';
      }
    } catch (error) {
      // Cross-origin restriction; CSS min-height will be used.
    }
  };

  /**
   * Handles tab click events.
   *
   * @param {Event} event - The click event.
   * @param {HTMLElement} clickedTab - The clicked tab element.
   */
  EmailTemplatePreview.prototype.handleTabClick = function (event, clickedTab) {
    event.preventDefault();
    event.stopPropagation();

    const targetTab = clickedTab.getAttribute(DATA_ATTRS.TAB);

    this.setActiveTab(clickedTab);
    this.setActiveContent(targetTab);

    if (targetTab === TABS.PREVIEW) {
      this.loadPreview();
    }
  };

  /**
   * Sets the active tab.
   *
   * @param {HTMLElement} activeTab - The tab to set as active.
   */
  EmailTemplatePreview.prototype.setActiveTab = function (activeTab) {
    this.tabs.forEach(function (tab) {
      tab.classList.remove(CLASSES.TAB_ACTIVE);
    });

    activeTab.classList.add(CLASSES.TAB_ACTIVE);
  };

  /**
   * Sets the active content panel.
   *
   * @param {string} targetTab - The tab identifier to activate.
   */
  EmailTemplatePreview.prototype.setActiveContent = function (targetTab) {
    this.contents.forEach(function (content) {
      const contentType = content.getAttribute(DATA_ATTRS.CONTENT);

      if (contentType === targetTab) {
        content.classList.add(CLASSES.TAB_CONTENT_ACTIVE);
      } else {
        content.classList.remove(CLASSES.TAB_CONTENT_ACTIVE);
      }
    });
  };

  /**
   * Loads the email preview into the iframe.
   *
   * Constructs the preview URL with body and subject parameters
   * from the form fields and sets it as the iframe source.
   */
  EmailTemplatePreview.prototype.loadPreview = function () {
    const bodyInput = this.getFormField(this.bodyFieldName);

    if (!bodyInput) {
      return;
    }

    const url = new URL(this.previewUrl, window.location.origin);
    url.searchParams.set('body', bodyInput.value);

    if (this.subjectFieldName) {
      const subjectInput = this.getFormField(this.subjectFieldName);

      if (subjectInput) {
        url.searchParams.set('subject', subjectInput.value);
      }
    }

    this.iframe.src = url.toString();
  };

  /**
   * Gets a form field element by name.
   *
   * @param {string} fieldName - The name attribute of the field.
   * @return {HTMLElement|null} The form field element or null.
   */
  EmailTemplatePreview.prototype.getFormField = function (fieldName) {
    return this.container.querySelector('[name="' + fieldName + '"]');
  };

  /**
   * Drupal behavior for email template preview.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the email template preview functionality to each tabs container.
   */
  Drupal.behaviors.emailTemplatePreview = {
    attach: function (context) {
      const previewUrl = drupalSettings.emailTemplatePreview?.url;

      if (!previewUrl) {
        return;
      }

      once(
        'email-template-tabs',
        '.' + CLASSES.TABS_CONTAINER,
        context
      ).forEach(function (container) {
        new EmailTemplatePreview(container, previewUrl);
      });
    },
  };
})(Drupal, drupalSettings, once);
