<?php

declare(strict_types=1);

namespace Drupal\user_email_preview\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Utility\Token;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for rendering email template previews.
 */
class EmailPreviewController extends ControllerBase {

  /**
   * Constructs an EmailPreviewController object.
   *
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(
    protected Token $token,
    protected RendererInterface $renderer,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('token'),
      $container->get('renderer')
    );
  }

  /**
   * Renders an email template preview.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rendered HTML response.
   */
  public function preview(Request $request): Response {
    $body = $request->query->get('body', '');
    $subject = $request->query->get('subject', '');

    // Sanitize input to prevent XSS attacks while preserving inline styles.
    // This route requires 'administer account settings' permission,
    // so only trusted admins can access it.
    $body = $this->sanitizeHtml($body);
    $subject = htmlspecialchars($subject, ENT_QUOTES, 'UTF-8');

    // Get sample user for token replacement.
    $current_user = $this->currentUser();
    $user = $this->entityTypeManager()->getStorage('user')->load($current_user->id());

    // Replace tokens in body and subject.
    $token_data = ['user' => $user];
    $token_options = ['clear' => TRUE];

    $rendered_body = $this->token->replace($body, $token_data, $token_options);
    $rendered_subject = strip_tags($this->token->replace($subject, $token_data, $token_options));

    // If the content is plain text (no HTML tags), apply basic formatting.
    if (!$this->containsHtmlTags($rendered_body)) {
      $rendered_body = $this->formatPlainText($rendered_body);
    }

    // Get site settings for the "from" field.
    $from_email = $this->config('system.site')->get('mail');
    $from_name = $this->config('system.site')->get('name');

    // Build the HTML response using the theme system.
    $build = [
      '#theme' => 'user_email_preview',
      '#subject' => $rendered_subject,
      '#from_name' => $from_name,
      '#from_email' => $from_email,
      '#to_email' => $user->getEmail(),
      '#body' => $rendered_body,
    ];
    $html = (string) $this->renderer->renderInIsolation($build);

    return new Response($html, 200, [
      'Content-Type' => 'text/html; charset=UTF-8',
      // Prevent the preview from being embedded elsewhere.
      'X-Frame-Options' => 'SAMEORIGIN',
      // Prevent MIME type sniffing.
      'X-Content-Type-Options' => 'nosniff',
    ]);
  }

  /**
   * Sanitizes HTML while preserving inline styles.
   *
   * Removes script tags and JavaScript event handlers while keeping
   * style attributes needed for email template rendering.
   *
   * @param string $html
   *   The HTML to sanitize.
   *
   * @return string
   *   The sanitized HTML.
   */
  protected function sanitizeHtml(string $html): string {
    // Remove script tags and their contents.
    $html = preg_replace('/<script\b[^>]*>.*?<\/script>/is', '', $html);

    // Remove event handler attributes (onclick, onerror, onload, etc.).
    $html = preg_replace('/\s+on\w+\s*=\s*["\'][^"\']*["\']/i', '', $html);
    $html = preg_replace('/\s+on\w+\s*=\s*[^\s>]+/i', '', $html);

    // Remove javascript: URLs from href and src attributes.
    $html = preg_replace('/(<[^>]+\s)(href|src)\s*=\s*["\']?\s*javascript:[^"\'>\\s]*/i', '$1$2=""', $html);

    return $html;
  }

  /**
   * Checks if a string contains HTML tags.
   *
   * @param string $text
   *   The text to check.
   *
   * @return bool
   *   TRUE if the text contains HTML tags, FALSE otherwise.
   */
  protected function containsHtmlTags(string $text): bool {
    // Check for common HTML tags (excluding self-closing tags like <br>).
    return $text !== strip_tags($text);
  }

  /**
   * Formats plain text for HTML display using Drupal's text format system.
   *
   * @param string $text
   *   The plain text to format.
   *
   * @return string
   *   The formatted HTML.
   */
  protected function formatPlainText(string $text): string {
    $build = [
      '#type' => 'processed_text',
      '#text' => $text,
      '#format' => 'plain_text',
    ];

    return (string) $this->renderer->renderInIsolation($build);
  }

}
