<?php

declare(strict_types=1);

namespace Drupal\user_email_preview\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * Form alter and theme hooks for email template preview.
 */
class UserAdminSettingsHooks {

  use StringTranslationTrait;

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    return [
      'user_email_preview' => [
        'variables' => [
          'subject' => '',
          'from_name' => '',
          'from_email' => '',
          'to_email' => '',
          'body' => '',
        ],
        'template' => 'user-email-preview',
      ],
    ];
  }

  /**
   * Implements hook_form_FORM_ID_alter() for user_admin_settings.
   *
   * Using Order::Last to ensure this runs after other modules
   * that may add additional email template sections.
   */
  #[Hook('form_user_admin_settings_alter', order: Order::Last)]
  public function formAlter(array &$form, FormStateInterface $form_state): void {
    // Attach the library for styling and JavaScript.
    $form['#attached']['library'][] = 'user_email_preview/email_template_preview';

    // Pass the preview URL to JavaScript.
    $preview_url = Url::fromRoute('user_email_preview.preview')->toString();
    $form['#attached']['drupalSettings']['emailTemplatePreview']['url'] = $preview_url;

    // Find and process all email template sections.
    $email_templates = $this->detectEmailTemplates($form);

    foreach ($email_templates as $template) {
      $this->wrapWithTabs($form, $template);
    }
  }

  /**
   * Detects email template sections in the form.
   *
   * @param array $form
   *   The form array.
   *
   * @return array
   *   Array of detected email templates with their field information.
   */
  protected function detectEmailTemplates(array $form): array {
    $templates = [];

    foreach ($form as $key => $element) {
      // Skip non-array elements and properties.
      if (!is_array($element) || str_starts_with($key, '#')) {
        continue;
      }

      // Check if this is an email template section (part of 'email' group).
      if (!isset($element['#group']) || $element['#group'] !== 'email') {
        continue;
      }

      // Look for body and subject fields in the element or nested containers.
      $fields = $this->findEmailFields($element, $key);

      if ($fields['body']) {
        $templates[] = [
          'section_key' => $key,
          'body_field' => $fields['body']['name'],
          'subject_field' => $fields['subject']['name'] ?? NULL,
          'container' => $fields['body']['container'],
        ];
      }
    }

    return $templates;
  }

  /**
   * Finds the body and subject fields in an email template section.
   *
   * @param array $element
   *   The form element to search.
   * @param string $section_key
   *   The section key for generating unique IDs.
   *
   * @return array
   *   Array with 'body' and 'subject' keys containing field info.
   */
  protected function findEmailFields(array $element, string $section_key): array {
    $result = ['body' => NULL, 'subject' => NULL];

    // First, look for fields directly in the element.
    foreach ($element as $field_key => $field) {
      if (!is_array($field) || str_starts_with($field_key, '#')) {
        continue;
      }

      // Check if this is a textarea (body field).
      if (isset($field['#type']) && $field['#type'] === 'textarea') {
        $result['body'] = [
          'name' => $field_key,
          'container' => NULL,
        ];
      }

      // Check if this is a textfield with 'subject' in the name.
      if (isset($field['#type']) && $field['#type'] === 'textfield' && str_contains($field_key, 'subject')) {
        $result['subject'] = [
          'name' => $field_key,
          'container' => NULL,
        ];
      }
    }

    // Look in nested containers (like 'settings' container).
    foreach ($element as $container_key => $container) {
      if (!is_array($container) || str_starts_with($container_key, '#')) {
        continue;
      }

      if (isset($container['#type']) && $container['#type'] === 'container') {
        foreach ($container as $field_key => $field) {
          if (!is_array($field) || str_starts_with($field_key, '#')) {
            continue;
          }

          if (isset($field['#type']) && $field['#type'] === 'textarea') {
            $result['body'] = [
              'name' => $field_key,
              'container' => $container_key,
            ];
          }

          if (isset($field['#type']) && $field['#type'] === 'textfield' && str_contains($field_key, 'subject')) {
            $result['subject'] = [
              'name' => $field_key,
              'container' => $container_key,
            ];
          }
        }
      }
    }

    return $result;
  }

  /**
   * Wraps email template fields with tabs (Template/Preview).
   *
   * @param array &$form
   *   The form array.
   * @param array $template
   *   The template configuration.
   */
  protected function wrapWithTabs(array &$form, array $template): void {
    $section_key = $template['section_key'];
    $body_field = $template['body_field'];
    $subject_field = $template['subject_field'] ?? NULL;
    $container = $template['container'];

    if (!isset($form[$section_key])) {
      return;
    }

    // Determine where the fields are located.
    if ($container && isset($form[$section_key][$container])) {
      $target = &$form[$section_key][$container];
    }
    else {
      $target = &$form[$section_key];
    }

    // Generate unique IDs.
    $base_id = str_replace('_', '-', $section_key);
    $iframe_id = 'email-preview-iframe-' . $base_id;
    $tabs_id = 'email-tabs-' . $base_id;

    // Extract all form fields (excluding properties starting with #).
    $fields = [];
    foreach ($target as $key => $value) {
      if (!str_starts_with($key, '#')) {
        $fields[$key] = $value;
        unset($target[$key]);
      }
    }

    // Build data attributes for the tabs container.
    $data_attributes = [
      'class' => ['email-template-tabs'],
      'id' => $tabs_id,
      'data-body-field' => $body_field,
      'data-iframe-id' => $iframe_id,
    ];
    if ($subject_field) {
      $data_attributes['data-subject-field'] = $subject_field;
    }

    // Create tabs wrapper using Gin theme classes.
    $target['email_tabs_' . $section_key] = [
      '#type' => 'container',
      '#attributes' => $data_attributes,
      'tabs_wrapper' => [
        '#type' => 'html_tag',
        '#tag' => 'nav',
        '#attributes' => [
          'class' => ['tabs-wrapper', 'is-horizontal'],
        ],
        'tabs' => [
          '#type' => 'html_tag',
          '#tag' => 'ul',
          '#attributes' => [
            'class' => ['tabs', 'tabs--secondary'],
          ],
          'template_tab' => [
            '#type' => 'html_tag',
            '#tag' => 'li',
            '#attributes' => [
              'class' => ['tabs__tab'],
            ],
            'link' => [
              '#type' => 'html_tag',
              '#tag' => 'button',
              '#value' => $this->t('Template'),
              '#attributes' => [
                'type' => 'button',
                'class' => ['tabs__link', 'is-active'],
                'data-tab' => 'template',
              ],
            ],
          ],
          'preview_tab' => [
            '#type' => 'html_tag',
            '#tag' => 'li',
            '#attributes' => [
              'class' => ['tabs__tab'],
            ],
            'link' => [
              '#type' => 'html_tag',
              '#tag' => 'button',
              '#value' => $this->t('Preview'),
              '#attributes' => [
                'type' => 'button',
                'class' => ['tabs__link'],
                'data-tab' => 'preview',
              ],
            ],
          ],
        ],
      ],
      'template_content' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['email-tab-content', 'email-tab-content--active'],
          'data-content' => 'template',
        ],
      ] + $fields,
      'preview_content' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['email-tab-content'],
          'data-content' => 'preview',
        ],
        'iframe' => [
          '#type' => 'html_tag',
          '#tag' => 'iframe',
          '#attributes' => [
            'id' => $iframe_id,
            'class' => ['email-preview-iframe'],
            'src' => 'about:blank',
            'title' => $this->t('Email preview'),
          ],
        ],
      ],
    ];
  }

}
