<?php

namespace Drupal\user_export_csv_vcf\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller callback for generating and downloading CSV file.
 *
 * @param int $uid
 *   The user ID.
 *
 * @return \Symfony\Component\HttpFoundation\Response|\Symfony\Component\HttpFoundation\RedirectResponse
 */
class DownloadCsvController extends ControllerBase {

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a object.
   *
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(FileUrlGeneratorInterface $file_url_generator, EntityTypeManagerInterface $entity_type_manager) {
    $this->fileUrlGenerator = $file_url_generator;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('file_url_generator'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Controller callback for generating and downloading CSV file.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A response object that triggers the download of a CSV file.
   */
  public function downloadCsv($uid) {
    // Load the user entity.
    $user = $this->entityTypeManager->getStorage('user')->load($uid);

    if (!$user) {
      $this->messenger()->addError($this->t('User not found.'));
      return $this->redirect('<current>');
    }

    // Extract fields.
    $phone_number = $user->get('field_phone_number')->value;
    $email = $user->getEmail();
    $comment = $user->get('field_comment')->value ?? '';

    $firstname = $lastname = $organization = '';
    $full_address = '';
    $address_field = $user->get('field_address')->first();
    if ($address_field) {
      $address = $address_field->getValue();
      $firstname = $address['given_name'] ?? '';
      $lastname = $address['family_name'] ?? '';
      $organization = $address['organization'] ?? '';
      $address_parts = array_filter([
        $address['address_line1'] ?? '',
        $address['address_line2'] ?? '',
        $address['locality'] ?? '',
        $address['administrative_area'] ?? '',
        $address['postal_code'] ?? '',
        $address['country_code'] ?? '',
      ]);
      $full_address = implode(', ', $address_parts);
    }

    // Profile picture URL.
    $image_url = '';
    if ($user->hasField('user_picture') && !$user->get('user_picture')->isEmpty()) {
      $file = $user->get('user_picture')->entity;
      if ($file) {
        $image_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
      }
    }

    // print_r($image_url);
    // exit;
    // Validation
    $missingFields = [];
    if (empty($phone_number)) {
      $missingFields[] = 'Phone Number';
    }
    if (empty($email)) {
      $missingFields[] = 'Email';
    }
    if (empty($full_address)) {
      $missingFields[] = 'Address';
    }
    if (!empty($missingFields)) {
      $this->messenger()->addError($this->t('CSV could not be downloaded, the following fields are required: @fields', [
        '@fields' => implode(', ', $missingFields),
      ]));
      return $this->redirect('<current>');
    }

    // Prepare CSV content.
    $header = [
      'First Name',
      'Last Name',
      'Organization',
      'Phone Number',
      'Email',
      'Address',
      'Comment',
      'Profile Picture URL',
    ];
    $row = [$firstname, $lastname, $organization, $phone_number, $email, $full_address, $comment, $image_url];

    $handle = fopen('php://temp', 'r+');
    fputcsv($handle, $header);
    fputcsv($handle, $row);
    rewind($handle);
    $csv_content = stream_get_contents($handle);
    fclose($handle);

    // Return as download.
    $filename = str_replace(' ', '_', $firstname . '_' . $lastname) . '.csv';
    $response = new Response($csv_content);
    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    return $response;
  }

}
