<?php

namespace Drupal\user_export_csv_vcf\Plugin\Action;

use Symfony\Component\HttpFoundation\Response;
use Drupal\Core\Action\ActionBase;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a "Download selected users as CSV" action.
 *
 * @Action(
 *   id = "user_export_csv_vcf_download_user_csv",
 *   label = @Translation("Download selected users as CSV"),
 *   type = "user"
 * )
 */
class DownloadUserCsvAction extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * The file URL generator service.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, FileUrlGeneratorInterface $file_url_generator) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_url_generator')
    );
  }

  /**
   * Required by interface even if not used.
   */
  public function execute($entity = NULL) {
    // Do nothing here. We use executeMultiple().
  }

  /**
   * Execute action for multiple selected users.
   */
  public function executeMultiple(array $entities) {
    $rows = [];

    foreach ($entities as $user) {
      if ($user->isAnonymous()) {
        continue;
      }

      $email = $user->getEmail();
      $phone = $user->get('field_phone_number')->value ?? '';
      $comment = $user->get('field_comment')->value ?? '';
      $address = $user->get('field_address')->first()?->getValue() ?? [];

      $firstname = $address['given_name'] ?? '';
      $lastname = $address['family_name'] ?? '';
      $organization = $address['organization'] ?? '';
      $full_address = implode(', ', array_filter([
        $address['address_line1'] ?? '',
        $address['address_line2'] ?? '',
        $address['locality'] ?? '',
        $address['administrative_area'] ?? '',
        $address['postal_code'] ?? '',
        $address['country_code'] ?? '',
      ]));

      $image_url = '';
      if (!$user->get('user_picture')->isEmpty()) {
        $file = $user->get('user_picture')->entity;
        if ($file) {
          $image_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
        }
      }

      $rows[] = [
        $firstname,
        $lastname,
        $organization,
        $phone,
        $email,
        $full_address,
        $comment,
        $image_url,
      ];
    }

    // Create and output CSV.
    $header = [
      'First Name',
      'Last Name',
      'Organization',
      'Phone Number',
      'Email',
      'Address',
      'Comment',
      'Profile Picture URL',
    ];
    $handle = fopen('php://temp', 'r+');
    fputcsv($handle, $header);
    foreach ($rows as $row) {
      fputcsv($handle, $row);
    }
    rewind($handle);
    $csv = stream_get_contents($handle);
    fclose($handle);

    $response = new Response($csv);
    $filename = 'exported_users_' . date('Ymd_His') . '.csv';
    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    $response->send();
  }

  /**
   * Access control for the action.
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $object->access('view', $account, $return_as_object);
  }

}
