<?php

namespace Drupal\user_export_csv_vcf\Plugin\Action;

use Drupal\Core\Action\ActionBase;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\Response;

/**
 * Provides a "Download selected users as User Export CSV and VCF" action.
 *
 * @Action(
 *   id = "user_export_csv_vcf_download_user_export_vcf",
 *   label = @Translation("Download selected users as VCF"),
 *   type = "user"
 * )
 */
class DownloadUserVcfAction extends ActionBase {

  /**
   * Required by interface even if not used.
   */
  public function execute($entity = NULL) {
    // Required by ActionBase, even if not used.
  }

  /**
   * Execute action for multiple selected users.
   */
  public function executeMultiple(array $entities) {
    $vcards = [];

    foreach ($entities as $user) {
      if (!$user instanceof User || $user->isAnonymous()) {
        continue;
      }

      $email = $user->getEmail();
      $phone = $user->get('field_phone_number')->value ?? '';
      $comment = $user->get('field_comment')->value ?? '';
      $address = $user->get('field_address')->first()?->getValue() ?? [];
      $firstname = $address['given_name'] ?? '';
      $lastname = $address['family_name'] ?? '';
      $organization = $address['organization'] ?? '';
      $full_address = implode(', ', array_filter([
        $address['address_line1'] ?? '',
        $address['address_line2'] ?? '',
        $address['locality'] ?? '',
        $address['administrative_area'] ?? '',
        $address['postal_code'] ?? '',
        $address['country_code'] ?? '',
      ]));

      $base64_image = '';
      if (!$user->get('user_picture')->isEmpty()) {
        $file = $user->get('user_picture')->entity;
        if ($file) {
          $image_data = file_get_contents($file->getFileUri());
          $base64_image = base64_encode($image_data);
        }
      }

      $user_export_csv_vcf = [];
      $user_export_csv_vcf[] = 'BEGIN:VCARD';
      $user_export_csv_vcf[] = 'VERSION:3.0';
      $user_export_csv_vcf[] = 'N:' . $lastname . ';' . $firstname;
      $user_export_csv_vcf[] = 'FN:' . trim($firstname . ' ' . $lastname);
      if (!empty($organization)) {
        $user_export_csv_vcf[] = 'ORG:' . $organization;
      }
      $user_export_csv_vcf[] = 'TEL:' . $phone;
      $user_export_csv_vcf[] = 'EMAIL:' . $email;
      $user_export_csv_vcf[] = 'ADR:' . $full_address;
      if (!empty($comment)) {
        $user_export_csv_vcf[] = 'NOTE:' . str_replace(["\r", "\n"], ' ', $comment);
      }
      if (!empty($base64_image)) {
        $user_export_csv_vcf[] = 'PHOTO;ENCODING=b;TYPE=JPEG:' . $base64_image;
      }
      $user_export_csv_vcf[] = 'END:VCARD';

      $vcards[] = implode("\r\n", $user_export_csv_vcf);
    }

    $user_export_csv_vcfContent = implode("\r\n\r\n", $vcards);

    $filename = 'users_vcards_' . date('Ymd_His') . '.vcf';
    $response = new Response($user_export_csv_vcfContent);
    $response->headers->set('Content-Type', 'text/vcard');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    $response->send();
  }

  /**
   * Access control for the action.
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    return $object->access('view', $account, $return_as_object);
  }

}
