<?php

namespace Drupal\user_export_csv_vcf\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for handling the Download User Export CSV and VCF functionality.
 */
class DownloadVcfController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Controller callback for generating and downloading User Export CSV and VCF.
   *
   * @param int $uid
   *   The user ID.
   *
   * @return \Symfony\Component\HttpFoundation\Response|\Symfony\Component\HttpFoundation\RedirectResponse
   *   A response object that triggers the download of a VCF file.
   */
  public function downloadVcf($uid) {
    // Load the user entity.
    // $user = User::load($uid);
    $user = $this->entityTypeManager->getStorage('user')->load($uid);

    // Get data.
    $phone_number = $user->get('field_phone_number')->value;
    $email = $user->getEmail();
    $comment = $user->get('field_comment')->value ?? '';

    // Image.
    $image_base64 = '';
    if ($user->hasField('user_picture') && !$user->get('user_picture')->isEmpty()) {
      $file = $user->get('user_picture')->entity;
      if ($file) {
        $image_data = file_get_contents($file->getFileUri());
        $image_base64 = base64_encode($image_data);
      }
    }

    // Address.
    $firstname = $lastname = $organization = '';
    $full_address = '';
    $address_field = $user->get('field_address')->first();
    if ($address_field) {
      $address = $address_field->getValue();
      $firstname = $address['given_name'] ?? '';
      $lastname = $address['family_name'] ?? '';
      $organization = $address['organization'] ?? '';
      $address_parts = array_filter([
        $address['address_line1'] ?? '',
        $address['address_line2'] ?? '',
        $address['locality'] ?? '',
        $address['administrative_area'] ?? '',
        $address['postal_code'] ?? '',
        $address['country_code'] ?? '',
      ]);
      $full_address = implode(', ', $address_parts);
    }

    // Validation.
    $missingFields = [];
    if (empty($phone_number)) {
      $missingFields[] = 'Phone Number';
    }
    if (empty($email)) {
      $missingFields[] = 'Email';
    }
    if (empty($full_address)) {
      $missingFields[] = 'Address';
    }
    if (!empty($missingFields)) {
      $this->messenger()->addError($this->t('User Export VCF could not be downloaded, the following fields are required: @fields', [
        '@fields' => implode(', ', $missingFields),
      ]));
      return $this->redirect('<current>');
    }

    // Filename.
    $username = str_replace(' ', '_', $firstname . '_' . $lastname);

    // Generate User Export CSV and VCF.
    $user_export_csv_vcfContent = $this->userExportCsvVcfContent($firstname, $lastname, $organization, $phone_number, $email, $full_address, $comment, $image_base64);

    // Return file.
    $response = new Response($user_export_csv_vcfContent);
    $response->headers->set('Content-Type', 'text/vcard');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $username . '.vcf"');
    return $response;
  }

  /**
   * Generate User Export CSV and VCF content string.
   */
  protected function userExportCsvVcfContent($firstname, $lastname, $organization, $phone_number, $email, $address, $comment, $base64_image = '') {
    $user_export_csv_vcf = [];
    $user_export_csv_vcf[] = 'BEGIN:VCARD';
    $user_export_csv_vcf[] = 'VERSION:3.0';
    $user_export_csv_vcf[] = 'N:' . $lastname . ';' . $firstname;
    $user_export_csv_vcf[] = 'FN:' . trim($firstname . ' ' . $lastname);
    if (!empty($organization)) {
      $user_export_csv_vcf[] = 'ORG:' . $organization;
    }
    $user_export_csv_vcf[] = 'TEL:' . $phone_number;
    $user_export_csv_vcf[] = 'EMAIL:' . $email;
    $user_export_csv_vcf[] = 'ADR:' . $address;
    if (!empty($comment)) {
      $user_export_csv_vcf[] = 'NOTE:' . str_replace(["\r", "\n"], ' ', $comment);
    }
    if (!empty($base64_image)) {
      $user_export_csv_vcf[] = 'PHOTO;ENCODING=b;TYPE=JPEG:' . $base64_image;
    }
    $user_export_csv_vcf[] = 'END:VCARD';
    return implode("\r\n", $user_export_csv_vcf);
  }

}
