<?php

namespace Drupal\user_inactivity_revoke_roles\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * User inactivity settings form.
 */
class UserInactivitySettingsForm extends ConfigFormBase {

  /**
   * Constructs a UserInactivitySettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager, protected EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId(): string {
    return 'user_inactivity_settings_form';
  }

  /**
   * {@inheritDoc}
   */
  protected function getEditableConfigNames(): array {
    return ['user_inactivity_revoke_roles.settings'];
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['help'] = [
      '#type' => 'container',
      '#markup' => $this->t('Set global user revoke settings on this page'),
    ];

    // Reference table for seconds conversion.
    $form['time_reference_table'] = [
      '#type' => 'details',
      '#title' => $this->t('Time reference table'),
      '#description' => $this->t('Use this table as a reference for setting time durations in seconds.'),
      '#open' => FALSE,
    ];

    $form['time_reference_table']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Time Increment'),
        $this->t('Seconds'),
      ],
      '#rows' => [
        [$this->t('1 Hour'), $this->t('3600')],
        [$this->t('1 Day'), $this->t('86400')],
        [$this->t('2 Days'), $this->t('172800')],
        [$this->t('1 Week'), $this->t('604800')],
        [$this->t('1 Month (30 Days)'), $this->t('2592000')],
        [$this->t('90 Days'), $this->t('7776000')],
        [$this->t('1 Year (365 Days)'), $this->t('31536000')],
      ],
    ];

    $config = $this->config('user_inactivity_revoke_roles.settings');

    $form['revoke_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic role revocation'),
      '#default_value' => $config->get('revoke_enabled') ?? FALSE,
    ];

    $roles_config = $config->get('user_revoke_roles') ?: [];
    $user_roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $roles = [];

    foreach ($user_roles as $role) {
      $roles[$role->id()] = $role->get('label');
    }

    // Save the current roles for use in submit handler.
    $form['current_roles'] = [
      '#type' => 'value',
      '#value' => $roles,
    ];

    // Now show boxes for each role.
    $form['user_revoke_roles'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Account revocations settings by role'),
      '#description' => $this->t('Set inactivity revoke times for each user role. Enter 0 to disable revocations for a role.'),
      '#states' => [
        'visible' => [
          ':input[name="revoke_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $excluded_roles = [
      RoleInterface::ANONYMOUS_ID,
      RoleInterface::AUTHENTICATED_ID,
    ];
    foreach ($roles as $rid => $role) {
      if (in_array($rid, $excluded_roles, TRUE)) {
        continue;
      }

      $form['user_revoke_roles']['user_revoke_' . $rid] = [
        '#type' => 'number',
        '#title' => $this->t('Seconds of inactivity before revoking %role users', ['%role' => $role]),
        '#default_value' => $roles_config[$rid] ?? 0,
        '#min' => 0,
      ];
    }
    // Account revocations warning settings.
    $form['warnings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Account revocations warning settings'),
      '#states' => [
        'visible' => [
          ':input[name="revoke_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Enable or disable account revocations warnings.
    $form['warnings']['send_revocation_warnings'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send account revoke warning emails'),
      '#default_value' => $config->get('send_revocation_warnings') ?? TRUE,
      '#description' => $this->t('If enabled, account revocations warning emails will be sent to users, starting at the configured offset time before account roles are revoked.'),
    ];

    $form['warnings']['extra_settings'] = [
      '#type' => 'container',
      '#states' => [
        // Hide the additional settings when revocations warnings are disabled.
        'invisible' => [
          'input[name="send_revocation_warnings"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['warnings']['extra_settings']['frequency'] = [
      '#type' => 'number',
      '#title' => $this->t('Notification frequency time (in seconds)'),
      '#default_value' => $config->get('frequency') ?: 172800,
      '#description' => $this->t('Specify how often (in seconds) warning emails should be sent. For example, 86400 = 1 day.'),
      '#min' => 0,
    ];

    $form['warnings']['extra_settings']['offset'] = [
      '#type' => 'number',
      '#title' => $this->t('Warning offset time (in seconds)'),
      '#default_value' => $config->get('offset') ?: 604800,
      '#description' => $this->t('Specify how far in advance (in seconds) to start sending warnings before account roles are revoked. For example, 604800 = 7 days.'),
      '#min' => 0,
    ];

    // Account revocations warning email template.
    $form['warnings']['extra_settings']['mail'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Account revocations warning message template'),
    ];

    $form['warnings']['extra_settings']['mail']['revocation_warning_mail_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config->get('revocation_warning_mail.subject') ?? '',
      '#description' => $this->t('Subject line for account revocations warning email messages.'),
      '#maxlength' => 180,
    ];

    $form['warnings']['extra_settings']['mail']['revocation_warning_mail_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $config->get('revocation_warning_mail.body') ?? '',
      '#description' => $this->t('Body for account revocations warning email messages.'),
      '#rows' => 15,
    ];

    $form['warnings']['extra_settings']['mail']['help'] = [
      '#markup' => $this->t('Available token variables for use in the email are: [site:name], [site:url], [site:mail], [user:display-name], [user:account-name], [user:mail], [site:login-url], [site:url-brief], [user:edit-url], [user:one-time-login-url], [user:cancel-url], [user:uirr_role_removed]'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (!ctype_digit($form_state->getValue('frequency'))) {
      $form_state->setErrorByName('frequency', $this->t('Frequency time must be an integer.'));
    }

    if (!ctype_digit($form_state->getValue('offset'))) {
      $form_state->setErrorByName('offset', $this->t('Warning offset time must be an integer.'));
    }

    $excluded_roles = [
      RoleInterface::ANONYMOUS_ID,
      RoleInterface::AUTHENTICATED_ID,
    ];

    foreach ($form_state->getValue('current_roles') as $rid => $role) {
      if (in_array($rid, $excluded_roles, TRUE)) {
        continue;
      }

      if (!ctype_digit($form_state->getValue('user_revoke_' . $rid))) {
        $form_state->setErrorByName('user_revoke_' . $rid, $this->t('Inactivity period must be an integer.'));
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('user_inactivity_revoke_roles.settings')
      ->set('revoke_enabled', $form_state->getValue('revoke_enabled'));

    if (!empty($form_state->getValue('frequency'))) {
      $config->set('frequency', (int) $form_state->getValue('frequency'));
    }

    if (!empty($form_state->getValue('offset'))) {
      $config->set('offset', (int) $form_state->getValue('offset'));
    }

    // Insert the rows that were inserted.
    $roles = $config->get('user_revoke_roles') ?: [];
    foreach ($form_state->getValue('current_roles') as $rid => $role) {
      // Only save non-zero values.
      if (!is_null($form_state->getValue('user_revoke_' . $rid))) {
        $roles[$rid] = (int) $form_state->getValue('user_revoke_' . $rid);
      }
    }
    $config->set('user_revoke_roles', $roles);

    // The notification email.
    $config->set('send_revocation_warnings', $form_state->getValue('send_revocation_warnings'));
    $config->set('revocation_warning_mail.subject', $form_state->getValue('revocation_warning_mail_subject'));
    $config->set('revocation_warning_mail.body', $form_state->getValue('revocation_warning_mail_body'));

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
