<?php

namespace Drupal\user_login_tracker\Controller;

use Drupal\user\Entity\User;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;

/**
 * Controller for Login Tracker Report.
 */
class LoginController extends ControllerBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Creates a new LoginController.
   *
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter.
   */
  public function __construct(DateFormatterInterface $dateFormatter) {
    $this->dateFormatter = $dateFormatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('date.formatter'),
    );
  }

  /**
   * Displays the login report for the day.
   */
  public function report() {
    // Get current date.
    $current_date = $this->dateFormatter->format(time(), 'custom', 'Y-m-d');

    // Query the database to get all logins for today, including login time.
    $database = Database::getConnection();
    $query = $database->select('user_login_tracker', 'lt');
    $query->fields('lt', ['uid', 'login_time']);
    $query->condition('lt.login_date', $current_date);
    $query->orderBy('lt.login_time', 'ASC');
    $result = $query->execute();

    // Prepare rows for the table.
    $rows = [];
    $user_logins = [];

    // Group login times by user.
    foreach ($result as $row) {
      // Load the user entity based on the user ID.
      $user = User::load($row->uid);
      $username = $user ? $user->get('name')->value : $this->t('Unknown User');

      // Store the login time for each user.
      if (!isset($user_logins[$row->uid])) {
        $user_logins[$row->uid] = [
          'username' => $username,
          'logins' => [],
        ];
      }

      // Add each login time for the user.
      $user_logins[$row->uid]['logins'][] = $row->login_time;
    }

    // Create rows.
    foreach ($user_logins as $uid => $user_data) {
      // Concatenate login times.
      $logins = implode(', ', $user_data['logins']);
      // Count of logins for the user.
      $login_count = count($user_data['logins']);
      $rows[] = [
        'data' => [
          $user_data['username'],
      // Display login times.
          $logins,
      // Display login count.
          $login_count,
        ],
      ];
    }

    $header = [
      $this->t('Username'),
      $this->t('Login Times'),
      $this->t('Login Count'),
    ];

    // Return the table render array.
    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No logins found for today.'),
    ];
  }

}
