<?php

namespace Drupal\user_logout\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\SessionManagerInterface;
use Drupal\user\RoleInterface;
use Drupal\user_logout\UserLogoutBatch;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for user logout forms.
 */
class UserLogoutForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The session manager service.
   *
   * @var \Drupal\Core\Session\SessionManagerInterface
   */
  protected SessionManagerInterface $sessionManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);
    $instance->database = $container->get('database');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->sessionManager = $container->get('session_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'user_logout_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Log out'),
      '#options' => [
        'user' => $this->t('A specific user(s)'),
        'role' => $this->t('All users having a specific role(s)'),
        'all' => $this->t('All users'),
      ],
      '#default_value' => 'user',
    ];
    $form['user'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('User(s)'),
      '#target_type' => 'user',
      '#tags' => TRUE,
      '#selection_settings' => [
        'include_anonymous' => FALSE,
      ],
      '#states' => [
        'visible' => [
          ':input[name="mode"]' => ['value' => 'user'],
        ],
      ],
    ];
    $form['role'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Role(s)'),
      '#options' => $this->getRoles(),
      '#states' => [
        'visible' => [
          ':input[name="mode"]' => ['value' => 'role'],
        ],
      ],
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Log out'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();

    $uids = [];
    switch ($values['mode']) {
      case 'user':
        $uids = array_column($values['user'], 'target_id');
        break;

      case 'role':
        $uids = $this->entityTypeManager
          ->getStorage('user')
          ->getQuery()
          ->condition('roles', $values['role'], 'IN')
          ->accessCheck(FALSE)
          ->execute();
        break;
    }

    if ($values['mode'] !== 'all' && empty($uids)) {
      $this->messenger()->addWarning("There isn't any user we can log out.");
      return;
    }

    // Get selected user sessions.
    $query = $this->database->select('sessions', 's');
    $query->leftJoin('users_field_data', 'u', 'u.uid = s.uid');
    if (!empty($uids)) {
      $query->condition('s.uid', $uids, 'IN');
    }
    $query->fields('s', ['uid']);
    // Remove duplicated users.
    $query->groupBy('uid');
    $result = $query->execute()->fetchAllAssoc('uid');

    if (!empty($result)) {
      // Don't logout the current user.
      unset($result[$this->currentUser()->id()]);
      $logged_users = array_keys($result);
      $batch = [
        'operations' => [],
        'finished' => [UserLogoutBatch::class, 'finish'],
        'title' => $this->t('User logout'),
        'init_message' => $this->t('Starting user logout.'),
        'progress_message' => $this->t('Completed @current step of @total.'),
        'error_message' => $this->t('User logout has encountered an error.'),
      ];
      foreach ($logged_users as $uid) {
        $batch['operations'][] = [
          [UserLogoutBatch::class, 'process'],
          [$this->sessionManager, $uid],
        ];
      }

      batch_set($batch);
    }
    else {
      $this->messenger()->addStatus("There isn't any active session for the selected user(s).");
    }
  }

  /**
   * Gets the roles to display in this form.
   *
   * @return array
   *   An array of roles keyed by ID.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getRoles(): array {
    $roles = [];
    $excluded_roles = [
      RoleInterface::ANONYMOUS_ID,
      RoleInterface::AUTHENTICATED_ID,
    ];

    foreach ($this->entityTypeManager->getStorage('user_role')->loadMultiple() as $role) {
      if (!in_array($role->id(), $excluded_roles)) {
        $roles[$role->id()] = $role->label();
      }
    }

    return $roles;
  }

}
