<?php

namespace Drupal\user_mailman_register\Form;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * @file
 * Contains \Drupal\user_mailman_register\Form\AdminEditForm.
 */

/**
 * Provides a form to add or edit a User Mailman Register list.
 */
class AdminEditForm implements FormInterface {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_mailman_register_adminedit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, $form_state, $list = NULL) {
    if (!$list) {
      $list = (object) [
        'lid' => 0,
        'name' => '',
        'command' => '',
        'admin' => '',
        'web' => '',
        'webarch' => '',
        'webadmin' => '',
        'webpass' => '',
        'version' => 1,
        'user_invite' => 0,
        'user_notify' => 0,
        'user_admin_notify' => 0,
        'allow_unsubscribe' => 0,
        'allow_temp_disable' => 0,
        'allow_digest' => 0,
        'description' => '',
        'defaultnewreg' => 0,
        'role_sync' => 0,
      ];
    }
    else {
      $list = _user_mailman_register_get_list($list);
      $account =
      [
        'role' => 'administrator',
      ];
      if (!$list) {
        return new RedirectResponse(Url::fromRoute('admin/structure/user_mailman_register')->toString());
      }
      // If list exists, add a descriptive header to the form.
      $form['list_header'] = [
        '#theme' => 'table',
        '#header' => [
          ['data' => t('ID')],
          ['data' => t('Name')],
          ['data' => t('Admin URL')],
          ['data' => t('Status')],
        ],
        '#rows' => [
          [
            'lid' => $list->lid ? $list->lid : '-',
            'name' => $list->name,
            'command' => $list->webadmin,
            'status' => (_user_mailman_register_allowed_list($list, $account)) ? t('Enabled') : t('Disabled'),
          ],
        ],
      ];
    }
    // Mailman list managment.
    $url = Url::fromRoute('user_mailman_register.administer_settings');
    $form['mailman_admin'] = [
      '#type' => 'fieldset',
      '#title' => t('Mailman settings'),
      '#tree' => TRUE,
    ];
    $form['mailman_admin']['name'] = [
      '#title' => 'Mailing List name',
      '#default_value' => !empty($list->name) ? $list->name : '',
      '#type' => 'textfield',
      '#required' => TRUE,
    ];
    $form['mailman_admin']['web'] = [
      '#title' => t('Mailing list web address for users'),
      '#default_value' => !empty($list->web) ? $list->web : '',
      '#description' => t('http://... Leave empty if hidden.'),
      '#type' => 'textfield',
    ];
    $form['mailman_admin']['webarch'] = [
      '#title' => t('Mailing list web archive address for users'),
      '#default_value' => !empty($list->webarch) ? $list->webarch : '',
      '#description' => t('http://... Leave empty if hidden.'),
      '#type' => 'textfield',
    ];
    $form['mailman_admin']['webadmin'] = [
      '#title' => t('Admin web page'),
      '#default_value' => $list->webadmin,
      '#description' => t("Mailman admin web page. Example: https://www.mysite.com/cgi-bin/mailman/admin/mylist.<br>
      It's highly recommended to use the https protocol if your server supports it.<br>
      The url has to be reachable from your Drupal web server.<br>
      Set empty to disable this list."),
      '#type' => 'textfield',
      '#required' => FALSE,
    ];
    $form['mailman_admin']['webpass'] = [
      '#title' => t('Admin web password'),
      '#default_value' => $list->webpass,
      '#description' => t('Mailman administrator web password.'),
      '#type' => 'textfield',
      '#size' => 15,
    ];
    $form['mailman_admin']['version'] = [
      '#title' => t('Mailman version'),
      '#description' => t('Select your mailman software version (it should be displayed on the mailman web interface footer).
      Try different settings if you realize that digest and nomail subscription does not work correctly.'),
      '#type' => 'radios',
      '#required' => TRUE,
      '#default_value' => $list->version,
      '#options' => [t('2.1.9 or prior'), t('2.1.10 or later')],
    ];
    $form['mailman_admin']['user_invite'] = [
      '#title' => t('Only invite users'),
      '#default_value' => $list->user_invite,
      '#description' => t('Just invite new users instead of subscribing them?'),
      '#type' => 'checkbox',
    ];
    $form['mailman_admin']['user_notify'] = [
      '#title' => t('Notify users'),
      '#default_value' => $list->user_notify,
      '#description' => t('Send welcome messages to the new subscribee?'),
      '#type' => 'checkbox',
    ];
    $form['mailman_admin']['user_admin_notify'] = [
      '#title' => t('Notify mailman administrators'),
      '#default_value' => $list->user_admin_notify,
      '#description' => t('Send notifications of new subscription to the list owner?'),
      '#type' => 'checkbox',
    ];
    // Drupal list management.
    $form['drupal_admin'] = [
      '#type' => 'fieldset',
      '#title' => t('List interface settings'),
      '#tree' => TRUE,
    ];
    $form['drupal_admin']['role_sync'] = [
      '#title' => t('Sync with roles'),
      '#default_value' => $list->role_sync,
      '#description' => t('Subscribe and/or unsubscribe users to this list according to their current role and list permissions. This action will be performed everytime a user role is changed.'),
      '#type' => 'radios',
      '#required' => TRUE,
      '#options' => _user_mailman_register_sync_options(),
    ];
    $form['drupal_admin']['defaultnewreg'] = [
      '#type' => 'radios',
      '#title' => t('Default choice'),
      '#options' => [0 => t('No'), 1 => t('Yes')],
      '#default_value' => $list->defaultnewreg,
      '#required' => TRUE,
      '#description' => t('This will be the default choice in the user subscription form.'),
    ];
    $form['drupal_admin']['allow_unsubscribe'] = [
      '#title' => t('Allow unsubscribe'),
      '#default_value' => $list->allow_unsubscribe,
      '#description' => t('Allow users to unsubscribe from the list? Unchecked will set <b>Default choice</b> option to <em>Yes</em> and force subscription to this list.'),
      '#type' => 'checkbox',
    ];
    $form['drupal_admin']['allow_temp_disable'] = [
      '#title' => t('Allow temporary disable'),
      '#default_value' => $list->allow_temp_disable,
      '#description' => t('Allow users to temporarily disable delivery?'),
      '#type' => 'checkbox',
    ];
    $form['drupal_admin']['allow_digest'] = [
      '#title' => t('Allow digest'),
      '#default_value' => $list->allow_digest,
      '#description' => t('Allow users to subscribe in digest mode?'),
      '#type' => 'checkbox',
    ];
    $form['drupal_admin']['description'] = [
      '#title' => t('Description'),
      '#default_value' => $list->description,
      '#description' => t('A brief description of this mailing list. This text will be displayed in the subscriptions page.'),
      '#type' => 'textarea',
      '#required' => TRUE,
    ];
    $form['list'] = [
      '#type' => 'value',
      '#value' => $list,
    ];
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => t('Save'),
      '#suffix' => Link::fromTextAndUrl(t('Cancel'), $url)->toString(),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, $form_state) {
    $web = $form_state->getValue('mailman_admin')['web'];
    if (!empty($web) && !UrlHelper::isValid($web, TRUE)) {
      $form_state->setErrorByName('web', t('%web is not a valid url', ['%web' => $web]));
    }
    $webarch = $form_state->getValue('mailman_admin')['webarch'];
    if (!empty($webarch) && !UrlHelper::isValid($webarch, TRUE)) {
      $form_state->setErrorByName('webarch', t('%webarch is not a valid url', ['%webarch' => $webarch]));
    }
    $webadmin = $form_state->getValue('mailman_admin')['webadmin'];
    if (!empty($webadmin) && !UrlHelper::isValid($webadmin, TRUE)) {
      $form_state->setErrorByName('webadmin', t('%webadmin is not a valid url', ['%webadmin' => $webadmin]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, $form_state) {
    $webpass = $form_state->getValue('mailman_admin')['webpass'];
    $list = [
      'lid' => $form_state->getValue('list')->lid,
      'name' => $form_state->getValue('mailman_admin')['name'],
      'web' => $form_state->getValue('mailman_admin')['web'],
      'webarch' => $form_state->getValue('mailman_admin')['webarch'],
      'webadmin' => $form_state->getValue('mailman_admin')['webadmin'],
      'webpass' => $webpass,
      'version' => $form_state->getValue('mailman_admin')['version'],
      'user_invite' => $form_state->getValue('mailman_admin')['user_invite'],
      'user_notify' => $form_state->getValue('mailman_admin')['user_notify'],
      'user_admin_notify' => $form_state->getValue('mailman_admin')['user_admin_notify'],
      'allow_unsubscribe' => $form_state->getValue('drupal_admin')['allow_unsubscribe'],
      'allow_temp_disable' => $form_state->getValue('drupal_admin')['allow_temp_disable'],
      'allow_digest' => $form_state->getValue('drupal_admin')['allow_digest'],
      'description' => $form_state->getValue('drupal_admin')['description'],
      'defaultnewreg' => $form_state->getValue('drupal_admin')['defaultnewreg'],
      'role_sync' => $form_state->getValue('drupal_admin')['role_sync'],
    ];
    if (empty($webpass)) {
      $webpass = $form_state->getValue('list')->webpass;
    }
    if (isset($list['lid'])) {
      // Update existing mailing list record.
      if (\Drupal::database()->merge('mailman_lists')
        ->key('lid', $list['lid'])
        ->fields($list)
        ->execute()) {
        $action = t('updated');
        $watchdog = 'notice';
        $log = '%name list settings @action';
        \Drupal::logger('user_mailman_register')->$watchdog($log,
        ['%name' => $list['name'], '@action' => $action]);
      }
      else {
        $action = t('not updated');
        $watchdog = WATCHDOG_ERROR;
      }
    }
    else {
      // Create new mailing list record.
      unset($list['lid']);
      if (\Drupal::database()->insert('mailman_lists')
        ->fields($list)
        ->execute()) {
        $action = t('created');
        $watchdog = 'notice';
      }
      else {
        $action = t('not created');
        $watchdog = 'error';
      }
    }
    return new RedirectResponse('admin/structure/user_mailman_register');
  }

}
