<?php

namespace Drupal\user_mailman_register\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a form for deleting lists.
 */
class AdminDeleteForm extends ConfirmFormBase {

  /**
   * The LID of the list to delete.
   *
   * @var int
   */
  protected $list;

  /**
   * The name of the list to delete.
   *
   * @var int
   */
  protected $name;

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, string $list = NULL) {
    $this->lid = $list;
    $database = \Drupal::database();
    $query = $database->select('mailman_lists', 'ml')
      ->condition('ml.lid', $list, '=')
      ->fields('ml', ['lid', 'name']);
    $result = $query->execute();
    $this->name = $result->name;
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_mailman_register_delete';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('user_mailman_register.administer_lists');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return t('Approve');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    // Fetch the list name from database using the list ID from $this.
    $database = \Drupal::database();
    $query = $database->select('mailman_lists', 'ml');
    $query->condition('ml.lid', $this->lid, "=");
    $query->fields('ml', ['name']);
    $result = $query->execute();
    $name = $result->fetchField(0);
    return $this->t('Are you sure you want to delete mailing list %name? This action cannot be undone.', ['%name' => $name]);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, $form_state, $list = NULL) {
    $database = \Drupal::database();
    $query = $database->select('mailman_lists', 'ml');
    $query->condition('ml.lid', $this->lid, "=");
    $query->fields('ml', ['name']);
    $result = $query->execute();
    $name = $result->fetchField(0);
    if ($query = $database->delete('mailman_lists')
      ->condition('lid', $this->lid)
      ->execute()) {
      $log = 'Mailman list %name deleted';
      $this->messenger()->addMessage($this->t('The Mailman list %name has been deleted.', ['%name' => $name]));
      \Drupal::logger('user_mailman_register')->notice($log, ['%name' => $this->name]);
    }
    else {
      $log = '%name list deletion failed';
      $this->messenger()->addMessage($this->t('Deletion failed for Mailman list %name.', ['%name' => $name]));
      \Drupal::logger('user_mailman_register')->error($log, ['%name' => $this->name]);
    }

    if ($query = $database->delete('mailman_users')
      ->condition('lid', $this->lid)
      ->execute()) {
      $log = 'All subscribers to Mailman list %name deleted';
      $this->messenger()->addMessage($this->t('All subscriptions to the Mailman list %name have been deleted from the Drupal server. They remain subscribed on the Mailman server.', ['%name' => $name]));
      \Drupal::logger('user_mailman_register')->notice($log, ['%name' => $this->name]);
    }
    else {
      $log = '%name list subscribers deletion failed';
      $this->messenger()->addMessage($this->t('Deletion failed for Mailman list %name subscriptions. This may simply mean there were no subscriptions for this list.', ['%name' => $name]));
      \Drupal::logger('user_mailman_register')->error($log, ['%name' => $name]);
    }

  }

}
