<?php

namespace Drupal\user_mailman_register\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\Markup;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;

/**
 * @file
 * Contains \Drupal\user_mailman_register\Form\AdminSettingsForm.
 */

/**
 * Provides a form to configure User Mailman Register settings.
 */
class AdminSettingsForm extends ConfigFormBase {
  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'user_mailman_register.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_mailman_register_adminsettings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, $form_state) {
    $config = $this->config(static::SETTINGS);
    $perms = user_role_permissions([AccountInterface::ANONYMOUS_ROLE => AccountInterface::ANONYMOUS_ROLE]);
    if (isset($perms[AccountInterface::ANONYMOUS_ROLE]['access user_mailman_register'])) {
      $url = Url::fromRoute('user.admin_permissions');
      $form['anonymous_warning'] = [
        '#type' => 'markup',
        '#prefix' => '<div class="messages warning">',
        '#markup' => t('Warning: anonymous users have the ability to subscribe to mailing lists without registering. To prevent this, turn off permission "access user_mailman_register" for anonymous users in !permissions_url.', ['!permissions_url' => Link::fromTextAndUrl(t('anonymous user permissions'), $url . AccountInterface::ANONYMOUS_ROLE)]),
        '#suffix' => '</div>',
      ];
    }
    $form['user_mailman_register_checkinterval'] = [
      '#type' => 'textfield',
      '#title' => t('Mailman connection interval'),
      '#size' => 5,
      '#maxlength' => 5,
      '#required' => TRUE,
      '#default_value' => $config->get('user_mailman_register_checkinterval'),
      '#description' => t("How often (in minutes) the user subscription form can connect to the mailman server to check his/her subscription status. During this interval, subscription status will be retrieved from the User Mailman Register stored status of the last connection to the mailman server. Set to 0 to connect every time users check their subscription form."),
    ];
    $form['user_mailman_register_hourly_threshold'] = [
      '#type' => 'select',
      '#title' => t('Hourly threshold'),
      '#options' =>
      array_combine([1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 20, 30, 40, 50],
      [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 20, 30, 40, 50]),
      '#default_value' => $config->get('user_mailman_register_hourly_threshold'),
      '#description' => t('The maximum number of subscription form submissions a user can perform per hour and per list. This affects all submissions for anonymous users and list invitation submissions for every user.'),
    ];
    $form['user_mailman_register_required'] = [
      '#type' => 'checkbox',
      '#title' => t('A subscription is required'),
      '#default_value' => $config->get('user_mailman_register_required'),
      '#description' => t('If checked, user must subscribe to at least one mailing list.'),
    ];
    $form['user_mailman_register_registerform'] = [
      '#type' => 'checkbox',
      '#title' => t('Visible in user registration form'),
      '#default_value' => $config->get('user_mailman_register_registerform'),
      '#description' => t('If checked, a "Mailing Lists Subscriptions" section 
	  will be displayed in the new user registration form. You will also have 
	  to grant %permission to "anonymous user" (to register) and "authenticated
	  user" (to manage their own subscription) in the @permissions_url for each
	  applicable mailing list.', [
     '%permission' => '"can subscribe to 
	  <listname>"',
     '@permissions_url' =>
     Markup::create(Link::fromTextAndUrl(t('User Mailman Register 
	  permissions'), Url::fromRoute('user.admin_permissions', [], [
     'fragment'
     => 'module-user_mailman_register',
   ]))->toString()->getGeneratedLink()),
   ]),
    ];
    $form['user_mailman_register_ownform'] = [
      '#type' => 'checkbox',
      '#title' => t('Display in own separate category'),
      '#default_value' => $config->get('user_mailman_register_ownform'),
      '#description' => t('If checked, mailing list subscription is displayed in a separate category on the <em>My account</em> page, otherwise it will be displayed in the user account edit form.'),
    ];
    if (\Drupal::moduleHandler()->moduleExists('profile')) {
      $url = Url::fromRoute('entity.profile.collection');
      $form['user_mailman_register_profilename'] = [
        '#type' => 'textfield',
        '#title' => t('Profile module'),
        '#size' => 50,
        '#default_value' => $config->get('user_mailman_register_profilename'),
        '#description' => t('Optional. Enter the name(s) of the profile module field(s) from which to retrieve the Mailman member name for the user. You can enter multiple fields separated by commas; e.g.: profile_firstname, profile_lastname. Order matters; fields will be concatenated in the order listed here.'),
      ];
    }
    $form['user_mailman_register_mailonerrors'] = [
      '#type' => 'textfield',
      '#title' => t('Mail on errors'),
      '#size' => 50,
      '#default_value' => $config->get('user_mailman_register_mailonerrors'),
      '#description' => t('If not empty, a warning mail will be sent to these addresses when a user can not modify his subscriptions because of a mailman server connection failure. Multiple emails should be comma separated. Optional.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    if (!is_numeric($form_state->getValue('user_mailman_register_checkinterval'))) {
      $form_state->setErrorByName('user_mailman_register_checkinterval', 'Not a valid number.');
    }
    if (isset($form['user_mailman_register_profilename'])) {
      $profile_string = $form_state->getValue('user_mailman_register_profilename');
      if ($profile_string) {
        if (\Drupal::moduleHandler()->moduleExists('profile')) {
          $profile_array = explode(',', $profile_string);
          foreach ($profile_array as $profile_field) {
            $parts = explode(':', $profile_field);
            if (!count($parts)) {
              $form_state->setErrorByName('user_mailman_register_profilename', t('%name is not a valid profile name.', ['%name' => $profile_field]));
            }
            else {
              foreach ($parts as $part) {
                $schema = \Drupal::database()->schema();
                if (!$schema->tableExists('profile__' . $part)) {
                  $form_state->setErrorByName('user_mailman_register_profilename',
                  t('%name is not a valid profile field.',
                  ['%name' => $profile_string]));
                }
              }
            }
          }
        }
      }
    }
    $mail_string = $form_state->getValue('user_mailman_register_mailonerrors');
    if ($mail_string) {
      $mail_array = explode(',', $mail_string);
      foreach ($mail_array as $mail) {
        $mail = trim($mail);
        if (!\Drupal::service('email.validator')->isValid($mail)) {
          $form_state->setErrorByName('user_mailman_register_mailonerror', t('%mail is not a valid email address, or is already subscribed.', ['%mail' => $mail]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Retrieve the configuration.
    $this->config(static::SETTINGS)
      // Set the submitted configuration settings.
      ->set('anonymous_warning', $form_state->getValue('anonymous_warning'))
      ->set('user_mailman_register_checkinterval', $form_state->getValue('user_mailman_register_checkinterval'))
      ->set('user_mailman_register_hourly_threshold', $form_state->getValue('user_mailman_register_hourly_threshold'))
      ->set('user_mailman_register_required', $form_state->getValue('user_mailman_register_required'))
      ->set('user_mailman_register_registerform', $form_state->getValue('user_mailman_register_registerform'))
      ->set('user_mailman_register_ownform', $form_state->getValue('user_mailman_register_ownform'))
      ->set('user_mailman_register_profilename', $form_state->getValue('user_mailman_register_profilename'))
      ->set('user_mailman_register_mailonerrors', $form_state->getValue('user_mailman_register_mailonerrors'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
