<?php

namespace Drupal\user_mailman_register\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Url;
use Drupal\user\Entity\User;

/**
 * @file
 * Contains \Drupal\user_mailman_register\Form\JoinListForm.
 */

/**
 * Provides form showing user edit page or anonymous subscription form.
 *
 * @param $account
 *   the account we're trying to subscribe. NULL for anonymous.
 * @param $show_submit
 *   TRUE = display this as a standalone form with submission button, vs a
 *   subform.
 */
class JoinListForm implements FormInterface {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_mailman_register_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, $form_state, $show_submit = TRUE) {
    $form['user_mailman_register']['lists'] = [
      '#type' => 'fieldset',
      '#title' => t('Mailing Lists'),
      '#description' => t('Join a mailing list to receive or participate in its discussions via email.'),
      '#collapsible' => FALSE,
      '#attributes' => [
        'id' => 'user_mailman_register',
        'class' => ['umr_list_fieldset'],
      ],
    ];
    $lists = _user_mailman_register_get_lists();
    if (empty($lists)) {
      $form['user_mailman_register']['no_lists'] = [
        '#type' => 'markup',
        '#prefix' => '<p>',
        '#markup' => t('There are no mailing lists available.'),
        '#suffix' => '</p>',
      ];
      return $form;
    }
    $lists_available = FALSE;
    $account = \Drupal::currentUser();
    $uid = \Drupal::routeMatch()->getParameter('user');
    // Non-admins can't view pages of other users, naturally.
    if ($account->id() != $uid && !$account->hasPermission('administer user_mailman_register')) {
      return;
    }
    else {
      // Store the admin account and use it later to show the mod field.
      $current_account = $account;
      // Load the user account from the route parameter.
      if (isset($uid) && $uid > 0) {
        $account = User::load($uid);
      }
    }
    foreach ($lists as $list) {
      if (!_user_mailman_register_allowed_list($list, $account)) {
        continue;
      }
      $list_status = USER_MAILMAN_REGISTER_UNSUBSCRIBED;
      $lists_available = TRUE;
      $options = [];
      $tlid = trim($list['lid']);
      $list_id = 'list' . $tlid;
      $form['user_mailman_register']['lists'][$list_id] = [
        '#type' => 'fieldset',
        '#title' => Html::escape($list['name']),
        '#description' => check_markup($list['description']),
        '#collapsible' => TRUE,
        '#attributes' => ['class' => ['umr_list_' . $tlid]],
      ];
      if (isset($account) && $account->id() > 0) {
        $subscription = _user_mailman_register_get_remote_status($account, $list);
        if (isset($subscription['lmessage'])) {
          $form['user_mailman_register']['lists'][$list_id]['listmsg'] = [
            '#type' => 'markup',
            '#prefix' => '<div class="messages error">',
            '#markup' => Html::escape($subscription['lmessage']),
            '#suffix' => '</div>',
          ];
        }
        $list_status = $subscription['lstatus'];
      }
      if ($list_status == USER_MAILMAN_REGISTER_UNSUBSCRIBED) {
        $title = t('Subscribe to this list?');
        if ($list['allow_unsubscribe']) {
          $options[USER_MAILMAN_REGISTER_UNSUBSCRIBED] = t('No');
        }
        $options[USER_MAILMAN_REGISTER_SUBSCRIBED_NORMAL] = t('Yes');
        // Feeds specified default value into the form if allowed.
        if ($list['defaultnewreg'] || !$list['allow_unsubscribe']) {
          $list_status = USER_MAILMAN_REGISTER_SUBSCRIBED_NORMAL;
        }
      }
      else {
        $title = t('Subscription settings');
        if ($list['allow_unsubscribe']) {
          $options[USER_MAILMAN_REGISTER_DO_UNSUBSCRIBE] = t('Unsubscribe');
        }
        if ($list['allow_temp_disable'] || $list_status == USER_MAILMAN_REGISTER_SUBSCRIBED_DISABLED) {
          $options[USER_MAILMAN_REGISTER_SUBSCRIBED_DISABLED] = _user_mailman_register_t_status(USER_MAILMAN_REGISTER_SUBSCRIBED_DISABLED);
        }
        if ($list['allow_digest'] || $list_status == USER_MAILMAN_REGISTER_SUBSCRIBED_DIGEST) {
          $options[USER_MAILMAN_REGISTER_SUBSCRIBED_DIGEST] = _user_mailman_register_t_status(USER_MAILMAN_REGISTER_SUBSCRIBED_DIGEST);
        }
        $options[USER_MAILMAN_REGISTER_SUBSCRIBED_NORMAL] = _user_mailman_register_t_status(USER_MAILMAN_REGISTER_SUBSCRIBED_NORMAL);
        // Display the moderation checkbox.
        if ($current_account->hasPermission('administer user_mailman_register')) {
          $form['user_mailman_register']['lists'][$list_id]['mod' . $tlid] = [
            '#type' => 'checkbox',
            '#title' => t('Moderate'),
            '#default_value' => $subscription['lmod'],
            '#description' => t('If checked, member postings will be moderated.'),
            '#weight' => 1,
          ];
        }
      }
      $form['user_mailman_register']['lists'][$list_id]['options' . $tlid] = [
        '#type' => 'radios',
        '#title' => $title,
        '#options' => $options,
        '#default_value' => $list_status,
      ];
    }
    if (!$lists_available) {
      $form['user_mailman_register']['lists']['#description'] =
      t('No mailing lists are available.');
      $form['user_mailman_register']['lists']['#attributes']['class'][] = 'umr_list_nolists';
    }
    elseif ($show_submit) {
      if (!$account || $account->id() === 0) {
        // Anonymous users can subscribe (if permissions are set to allow them)
        // but need to enter a mailing address.
        $form['user_mailman_register']['lists']['reqaddress'] = [
          '#type' => 'textfield',
          '#title' => t('Email address'),
          '#size' => 50,
          '#weight' => -1,
          '#required' => FALSE,
        ];
      }
      $form['submit'] = [
        '#type' => 'submit',
        '#value' => t('Save'),
      ];
    }
    $form['mailman_account'] = [
      '#type' => 'value',
      '#value' => $account,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, $form_state) {
    // Validation of email address for anonymous user subscription request.
    if (isset($form['user_mailman_register']['lists']['reqaddress'])) {
      $req_address = $form_state->getValue('reqaddress');
      if (!\Drupal::service('email.validator')->isValid($req_address)) {
        $form_state->setErrorByName('reqaddress', t('"%reqaddress" is not a valid email address, or is already subscribed.',
          ['%reqaddress' => $req_address]));
      }
    }
    // Validation if at least one subscription is required.
    $account = \Drupal::currentUser();
    if (!\Drupal::state()->get('user_mailman_register_required') &&
    !$account->hasPermission('administer user_mailman_register')) {
      $no_lists = TRUE;
      $lists = _user_mailman_register_get_lists();
      if (!empty($lists)) {
        foreach ($lists as $list) {
          if (isset($form['user_mailman_register']['lists']['list' . trim($list['lid'])]['options' . trim($list['lid'])])) {
            $edit_list = $form['user_mailman_register']['lists']['list' . trim($list['lid'])]['options' . trim($list['lid'])];
          }
          // Check all allowed lists except for those
          // with the unsubscribe option disabled.
          if (isset($edit_list) && _user_mailman_register_allowed_list($list, $account) && $list['allow_unsubscribe']) {
            $no_lists = FALSE;
            if ($edit_list > USER_MAILMAN_REGISTER_UNSUBSCRIBED) {
              return TRUE;
            }
          }
        }
      }
      if (!$no_lists) {
        $form_state->setErrorByName('user_mailman_register', t('At least one mailing list subscription is required'));
      }
      return $no_lists;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, $form_state) {
    $account = \Drupal::currentUser();
    if (!$account || $account->id() === 0) {
      $user_mail = $form_state->getValue('reqaddress');
      _user_mailman_register_subscribe($account, $form_state->getValues(), FALSE, $user_mail);
    }
    else {
      $account = $form_state->getValue('mailman_account');
      $user_mail = FALSE;
      if (!$account || !is_numeric($account->id())) {
        return;
      }
      $reqaddress = $form_state->getValue('reqaddress');
      if (isset($reqaddress)) {
        $user_mail = $form_state->getValue('reqaddress');
      }
      _user_mailman_register_subscribe($account, $form_state->getValues(), FALSE, $user_mail);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function userMailmanRegisterPage() {
    $user = \Drupal::currentUser();
    if ($user->uid > 0) {
      if (\Drupal::state()->get('user_mailman_register_ownform', 1)) {
        $path = 'user/' . $user->uid . '/user_mailman_register';
      }
      else {
        $path = 'user/' . $user->uid . '/edit';
      }
      return new RedirectResponse(Url::fromRoute($path)->toString());
    }
    else {
      return \Drupal::formBuilder()->getForm('Drupal\user_mailman_register\Form\JoinListForm', $user, TRUE);
    }
  }

}
