<?php

declare(strict_types=1);

namespace Drupal\user_preference_login_redirect\Hook;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Hook implementations for form alterations.
 */
class FormHooks implements ContainerInjectionInterface {

  /**
   * Constructs a FormHooks object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   */
  public function __construct(
    protected AccountProxyInterface $currentUser,
    protected ConfigFactoryInterface $configFactory,
    protected UserDataInterface $userData,
    protected ModuleHandlerInterface $moduleHandler,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('current_user'),
      $container->get('config.factory'),
      $container->get('user.data'),
      $container->get('module_handler'),
    );
  }

  /**
   * Implements hook_form_FORM_ID_alter() for user_form.
   */
  #[Hook('form_user_form_alter')]
  public function formUserFormAlter(&$form, FormStateInterface $form_state, $form_id): void {
    /** @var \Drupal\user\UserInterface $user */
    $user = $form_state->getFormObject()->getEntity();

    // Only add the field if the user has permission.
    if (!$this->currentUser->hasPermission('select login redirect preference in user profile')) {
      return;
    }

    // Get available routes from configuration.
    $config = $this->configFactory->get('user_preference_login_redirect.settings');
    $available_routes_text = $config->get('available_routes');

    if (empty($available_routes_text)) {
      return;
    }

    $routes = $this->parseRoutes($available_routes_text);

    if (empty($routes)) {
      return;
    }

    // Get the current user's preference.
    $current_preference = $this->userData->get('user_preference_login_redirect', $user->id(), 'redirect_route');

    // If no preference is set, use the first route as default.
    if (empty($current_preference)) {
      $current_preference = array_key_first($routes);
    }

    $form['user_preference_login_redirect'] = [
      '#type' => 'details',
      '#title' => $config->get('fieldset_title'),
      '#open' => TRUE,
      '#weight' => 10,
    ];

    $form['user_preference_login_redirect']['redirect_route'] = [
      '#type' => 'select',
      '#title' => $config->get('field_title'),
      '#description' => $config->get('field_description'),
      '#options' => $routes,
      '#default_value' => $current_preference,
      '#required' => TRUE,
    ];

    $form['actions']['submit']['#submit'][] = [$this, 'userFormSubmit'];
  }

  /**
   * Custom submit handler for user form.
   */
  public function userFormSubmit($form, FormStateInterface $form_state): void {
    /** @var \Drupal\user\UserInterface $user */
    $user = $form_state->getFormObject()->getEntity();
    $redirect_route = $form_state->getValue('redirect_route');

    if (!empty($redirect_route)) {
      $this->userData->set('user_preference_login_redirect', $user->id(), 'redirect_route', $redirect_route);
    }
  }

  /**
   * Parse routes from configuration text.
   *
   * @param string $routes_text
   *   The routes text from configuration.
   *
   * @return array
   *   An array of routes keyed by route name with labels as values.
   */
  protected function parseRoutes(string $routes_text): array {
    $routes = [];
    $lines = array_filter(array_map('trim', explode("\n", $routes_text)));

    foreach ($lines as $line) {
      if (strpos($line, '|') !== FALSE) {
        [$route, $label] = array_map('trim', explode('|', $line, 2));
        if (!empty($route) && !empty($label)) {
          $routes[$route] = $label;
        }
      }
    }

    return $routes;
  }

}
