<?php

declare(strict_types=1);

namespace Drupal\user_preference_login_redirect\Hook;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Url;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Hook implementations for user-related operations.
 */
class UserHooks implements ContainerInjectionInterface {

  /**
   * Constructs a UserHooks object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected UserDataInterface $userData,
    protected ModuleHandlerInterface $moduleHandler,
    protected RequestStack $requestStack,
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('user.data'),
      $container->get('module_handler'),
      $container->get('request_stack'),
      $container->get('logger.factory'),
    );
  }

  /**
   * Implements hook_user_login().
   */
  #[Hook('user_login')]
  public function userLogin(UserInterface $account): void {
    $config = $this->configFactory->get('user_preference_login_redirect.settings');
    $respect_destination = $config->get('respect_destination_parameter') ?? TRUE;

    // Check if we should respect the destination parameter.
    if ($respect_destination) {
      $request = $this->requestStack->getCurrentRequest();
      if ($request && $request->query->has('destination')) {
        // Let Drupal handle the destination parameter.
        return;
      }
    }

    // Get the user's redirect preference.
    $redirect_route = $this->userData->get('user_preference_login_redirect', $account->id(), 'redirect_route');

    if (!empty($redirect_route)) {
      // Validate that the route still exists in configuration.
      $available_routes_text = $config->get('available_routes');
      $routes = $this->parseRoutes($available_routes_text);

      if (isset($routes[$redirect_route])) {
        $destination = $redirect_route;

        // If destination was set to NULL by an alter hook, don't redirect.
        if ($destination === NULL) {
          return;
        }

        // Set the redirect destination.
        $request = $this->requestStack->getCurrentRequest();

        if ($destination === '<front>') {
          $url = Url::fromRoute('<front>');
        }
        else {
          try {
            $url = Url::fromRoute($destination);
          }
          catch (\Exception $e) {
            // If route is invalid, don't redirect.
            $this->loggerFactory->get('user_preference_login_redirect')->warning('Invalid route: @route', ['@route' => $destination]);
            return;
          }
        }

        if ($request) {
          $request->query->set('destination', $url->toString());
        }
      }
    }
  }

  /**
   * Parse routes from configuration text.
   *
   * @param string $routes_text
   *   The routes text from configuration.
   *
   * @return array
   *   An array of routes keyed by route name with labels as values.
   */
  protected function parseRoutes(string $routes_text): array {
    $routes = [];
    $lines = array_filter(array_map('trim', explode("\n", $routes_text)));

    foreach ($lines as $line) {
      if (strpos($line, '|') !== FALSE) {
        [$route, $label] = array_map('trim', explode('|', $line, 2));
        if (!empty($route) && !empty($label)) {
          $routes[$route] = $label;
        }
      }
    }

    return $routes;
  }

}
