<?php

declare(strict_types=1);

namespace Drupal\Tests\user_preference_login_redirect\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\user_preference_login_redirect\Hook\FormHooks;

/**
 * Tests the route parsing functionality.
 *
 * @group user_preference_login_redirect
 * @coversDefaultClass \Drupal\user_preference_login_redirect\Hook\FormHooks
 */
class RouteParserTest extends UnitTestCase {

  /**
   * The FormHooks instance.
   *
   * @var \Drupal\user_preference_login_redirect\Hook\FormHooks
   */
  protected $formHooks;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks for the dependencies.
    $current_user = $this->createMock('Drupal\Core\Session\AccountProxyInterface');
    $config_factory = $this->createMock('Drupal\Core\Config\ConfigFactoryInterface');
    $user_data = $this->createMock('Drupal\user\UserDataInterface');
    $module_handler = $this->createMock('Drupal\Core\Extension\ModuleHandlerInterface');

    $this->formHooks = new FormHooks(
      $current_user,
      $config_factory,
      $user_data,
      $module_handler
    );
  }

  /**
   * Tests parsing valid route configuration.
   *
   * @covers ::parseRoutes
   */
  public function testParseValidRoutes(): void {
    $input = "<front>|Default (Frontpage)\nuser.page|My Account\nnode.add|Create Content";

    $result = $this->invokeMethod($this->formHooks, 'parseRoutes', [$input]);

    $expected = [
      '<front>' => 'Default (Frontpage)',
      'user.page' => 'My Account',
      'node.add' => 'Create Content',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing routes with extra whitespace.
   *
   * @covers ::parseRoutes
   */
  public function testParseRoutesWithWhitespace(): void {
    $input = "  <front>  |  Default (Frontpage)  \n  user.page  |  My Account  ";

    $result = $this->invokeMethod($this->formHooks, 'parseRoutes', [$input]);

    $expected = [
      '<front>' => 'Default (Frontpage)',
      'user.page' => 'My Account',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing routes with empty lines.
   *
   * @covers ::parseRoutes
   */
  public function testParseRoutesWithEmptyLines(): void {
    $input = "<front>|Default (Frontpage)\n\nuser.page|My Account\n\n";

    $result = $this->invokeMethod($this->formHooks, 'parseRoutes', [$input]);

    $expected = [
      '<front>' => 'Default (Frontpage)',
      'user.page' => 'My Account',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing invalid routes.
   *
   * @covers ::parseRoutes
   */
  public function testParseInvalidRoutes(): void {
    $input = "invalid_route_without_pipe\n<front>|Valid Route";

    $result = $this->invokeMethod($this->formHooks, 'parseRoutes', [$input]);

    // Only the valid route should be parsed.
    $expected = [
      '<front>' => 'Valid Route',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing empty string.
   *
   * @covers ::parseRoutes
   */
  public function testParseEmptyString(): void {
    $input = '';

    $result = $this->invokeMethod($this->formHooks, 'parseRoutes', [$input]);

    $this->assertEmpty($result);
  }

  /**
   * Helper method to invoke protected/private methods.
   *
   * @param object $object
   *   The object instance.
   * @param string $method_name
   *   The method name to invoke.
   * @param array $parameters
   *   The parameters to pass to the method.
   *
   * @return mixed
   *   The method result.
   */
  protected function invokeMethod($object, string $method_name, array $parameters = []) {
    $reflection = new \ReflectionClass(get_class($object));
    $method = $reflection->getMethod($method_name);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

}
