<?php

declare(strict_types=1);

namespace Drupal\user_preference_login_redirect\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\user\UserDataInterface;
use Drupal\user_preference_login_redirect\RouteParser;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * Contains user preference login redirect drush commands.
 */
final class UserPreferenceLoginRedirectCommands extends DrushCommands {

  /**
   * Constructs a UserPreferenceLoginRedirectCommands object.
   */
  public function __construct(
    private readonly UserDataInterface $userData,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly RouteParser $routeParser,
  ) {
    parent::__construct();
  }

  /**
   * Bulk updates all users to use a given route.
   */
  #[CLI\Command(name: 'user_preference_login_redirect:bulk-update', aliases: ['uplr-bu'])]
  #[CLI\Argument(name: 'route', description: 'The route key or label to set for all users.')]
  #[CLI\Usage(name: 'uplr-bu <front>', description: 'Set all users to redirect to <front>')]
  #[CLI\Usage(name: 'uplr-bu "My Dashboard"', description: 'Set all users to redirect to the route labeled "My Dashboard"')]
  public function bulkUpdate(string $route): void {
    $config = $this->configFactory->get('user_preference_login_redirect.settings');
    $available_routes_text = $config->get('available_routes');

    if (empty($available_routes_text)) {
      $this->io()->error(dt('No available routes configured in user_preference_login_redirect.settings.'));
      return;
    }

    $routes = $this->routeParser->parseRoutes($available_routes_text);
    $route_key = NULL;

    // Check if the input matches a key.
    if (array_key_exists($route, $routes)) {
      $route_key = $route;
    }
    // Check if the input matches a label.
    elseif ($key = array_search($route, $routes)) {
      $route_key = $key;
    }

    if (!$route_key) {
      $this->io()->error(dt('The route / route label "@route" was not found in the configured available routes.', ['@route' => $route]));
      $this->io()->info(dt('Available routes:'));
      foreach ($routes as $key => $label) {
        $this->io()->info(dt('  @key (@label)', ['@key' => $key, '@label' => $label]));
      }
      return;
    }

    $this->io()->title(dt('Bulk updating users redirect preference'));
    $this->io()->text(dt('Target route: @key (@label)', ['@key' => $route_key, '@label' => $routes[$route_key]]));

    if (!$this->io()->confirm(dt('Are you sure you want to update ALL users?'))) {
      return;
    }

    $user_storage = $this->entityTypeManager->getStorage('user');
    $uids = $user_storage->getQuery()
      ->accessCheck(FALSE)
      ->execute();

    if (empty($uids)) {
      $this->io()->warning(dt('No users found.'));
      return;
    }

    $count = count($uids);
    $this->io()->progressStart($count);

    foreach ($uids as $uid) {
      $this->userData->set('user_preference_login_redirect', (int) $uid, 'redirect_route', $route_key);
      $this->io()->progressAdvance();
    }

    $this->io()->progressFinish();
    $this->io()->success(dt(
      'Updated @count users to use route "@route".',
      ['@count' => $count, '@route' => $route_key]
    ));
  }

}
