<?php

declare(strict_types=1);

namespace Drupal\user_preference_login_redirect\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;
use Drupal\user_preference_login_redirect\RouteParser;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Hook implementations for user-related operations and form alterations.
 */
class UserHooks implements ContainerInjectionInterface {

  /**
   * Constructs a UserHooks object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\user\UserDataInterface $userData
   *   The user data service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\user_preference_login_redirect\RouteParser $routeParser
   *   The route parser.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected UserDataInterface $userData,
    protected ModuleHandlerInterface $moduleHandler,
    protected RequestStack $requestStack,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected AccountProxyInterface $currentUser,
    protected RouteParser $routeParser,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('user.data'),
      $container->get('module_handler'),
      $container->get('request_stack'),
      $container->get('logger.factory'),
      $container->get('current_user'),
      $container->get('user_preference_login_redirect.route_parser'),
    );
  }

  /**
   * Implements hook_user_login().
   */
  #[Hook('user_login')]
  public function userLogin(UserInterface $account): void {
    $config = $this->configFactory->get('user_preference_login_redirect.settings');
    $respect_destination = $config->get('respect_destination_parameter') ?? TRUE;

    // Check if we should respect the destination parameter.
    if ($respect_destination) {
      $request = $this->requestStack->getCurrentRequest();
      if ($request && $request->query->has('destination')) {
        // Let Drupal handle the destination parameter.
        return;
      }
    }

    // Get the user's redirect preference.
    $redirect_route = $this->userData->get('user_preference_login_redirect', $account->id(), 'redirect_route');

    // The route is empty for some reason, return early:
    if (empty($redirect_route)) {
      return;
    }

    $route_with_parameters = explode('/', $redirect_route);
    // The first array entry should always be the original route:
    $route = $route_with_parameters[0];
    // Remove the route from the array:
    array_shift($route_with_parameters);
    $route_parameters = [];
    foreach ($route_with_parameters as $parameter) {
      [$entity_name, $entity_id] = explode(':', $parameter, 2);
      if (empty($entity_name) || empty($entity_id)) {
        continue;
      }
      $route_parameters[$entity_name] = $entity_id;
    }

    try {
      if (!empty($route_parameters)) {
        $url = Url::fromRoute($route, $route_parameters);
      }
      else {
        $url = Url::fromRoute($route);
      }
    }
    catch (\Exception $e) {
      // If route is invalid, don't redirect.
      $this->loggerFactory->get('user_preference_login_redirect')->warning('Invalid route: @route', ['@route' => $redirect_route]);
      return;
    }

    $this->requestStack->getCurrentRequest()?->query->set('destination', $url->toString());
  }

  /**
   * Implements hook_form_FORM_ID_alter() for user_form.
   */
  #[Hook('form_user_form_alter')]
  public function formUserFormAlter(&$form, FormStateInterface $form_state, $form_id): void {
    /** @var \Drupal\user\UserInterface $user */
    $user = $form_state->getFormObject()->getEntity();

    // Only add the field if the user has permission.
    if (!$this->currentUser->hasPermission('select login redirect preference in user profile')) {
      return;
    }

    // Get available routes from configuration.
    $config = $this->configFactory->get('user_preference_login_redirect.settings');
    $available_routes_text = $config->get('available_routes');

    if (empty($available_routes_text)) {
      return;
    }

    $routes = $this->routeParser->parseRoutes($available_routes_text);

    if (empty($routes)) {
      return;
    }

    // Get the current user's preference.
    $current_preference = $this->userData->get('user_preference_login_redirect', $user->id(), 'redirect_route');

    // If no preference is set, use the first route as default.
    if (empty($current_preference)) {
      $current_preference = array_key_first($routes);
    }

    $form['user_preference_login_redirect'] = [
      '#type' => 'details',
      '#title' => $config->get('fieldset_title'),
      '#open' => TRUE,
      '#weight' => $config->get('fieldset_weight') ?? 10,
    ];

    $form['user_preference_login_redirect']['redirect_route'] = [
      '#type' => 'select',
      '#title' => $config->get('field_title'),
      '#description' => $config->get('field_description'),
      '#options' => $routes,
      '#default_value' => $current_preference,
      '#required' => TRUE,
    ];

    $form['actions']['submit']['#submit'][] = [$this, 'userFormSubmit'];
  }

  /**
   * Custom submit handler for user form.
   */
  public function userFormSubmit($form, FormStateInterface $form_state): void {
    /** @var \Drupal\user\UserInterface $user */
    $user = $form_state->getFormObject()->getEntity();
    $redirect_route = $form_state->getValue('redirect_route');

    if (!empty($redirect_route)) {
      $this->userData->set('user_preference_login_redirect', $user->id(), 'redirect_route', $redirect_route);
    }
  }

}
