<?php

declare(strict_types=1);

namespace Drupal\user_preference_login_redirect\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Configure User preference login redirect settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'user_preference_login_redirect_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['user_preference_login_redirect.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('user_preference_login_redirect.settings');

    $form['available_routes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Available redirect routes'),
      '#description' => $this->t('Enter one route (with parameters) and label pair per line, separated by a pipe (|). The first route will be used as the default if no preference is set.<br>Examples:<ul><li>&ltfront&gt|Frontpage</li><li>system.performance_settings|Performance</li><li>entity.node.canonical/node:1/another_param:456|Overview</li></ul>'),
      '#default_value' => $config->get('available_routes'),
      '#required' => TRUE,
      '#rows' => 10,
    ];

    $form['respect_destination_parameter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Respect ?destination URL-parameter'),
      '#description' => $this->t('If enabled and a ?destination URL parameter is present, the redirect will not be executed and the regular destination redirect will be used instead.'),
      '#default_value' => $config->get('respect_destination_parameter') ?? TRUE,
    ];

    $form['fieldset_weight'] = [
      '#type' => 'number',
      '#title' => $this->t('Fieldset weight'),
      '#description' => $this->t('The weight of the fieldset in the user edit form. Default: 10'),
      '#default_value' => $config->get('fieldset_weight') ?? 10,
      '#required' => TRUE,
      '#step' => 1,
    ];

    $form['fieldset_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Fieldset title'),
      '#description' => $this->t('The title for the login redirect preference fieldset on the user edit form.'),
      '#default_value' => $config->get('fieldset_title') ?? $this->t('Login redirect preference'),
      '#required' => TRUE,
    ];

    $form['field_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Field title'),
      '#description' => $this->t('The title for the redirect route select field on the user edit form.'),
      '#default_value' => $config->get('field_title') ?? $this->t('Preferred login redirect destination'),
      '#required' => TRUE,
    ];

    $form['field_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Field description'),
      '#description' => $this->t('The description for the redirect route select field on the user edit form.'),
      '#default_value' => $config->get('field_description') ?? $this->t('Select where you would like to be redirected after logging in.'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $routes = $form_state->getValue('available_routes');
    $lines = array_filter(array_map('trim', explode("\n", $routes)));
    if (empty($lines)) {
      $form_state->setErrorByName('available_routes', $this->t('At least one route must be defined.'));
      return;
    }

    foreach ($lines as $line_number => $line) {
      if (strpos($line, '|') === FALSE) {
        $form_state->setErrorByName('available_routes', $this->t('Line @line is invalid. Each line must contain a route and label separated by a pipe (|).', [
          '@line' => $line_number + 1,
        ]));
      }
      else {
        $line_array = array_filter(array_map('trim', explode("|", $line)));
        $label = $line_array[1] ?? '';
        $route = $line_array[0] ?? '';
        if (empty($label)) {
          $form_state->setErrorByName('available_routes', $this->t('Line @line is invalid. Label may not be empty.', [
            '@line' => $line_number + 1,
          ]));
        }
        if (empty($route)) {
          $form_state->setErrorByName('available_routes', $this->t('Line @line is invalid. Route may not be empty.', [
            '@line' => $line_number + 1,
          ]));
        }
        else {
          // Check route validity:
          try {
            // @todo Add route parameters, if given
            Url::fromRoute($route);
          }
          catch (\Exception $e) {
            // If route is invalid, don't redirect.
            $form_state->setErrorByName('available_routes', $this->t('Line @line is invalid. Invalid route: %route', [
              '@line' => $line_number + 1,
              '%route' => $route,
            ]));
          }
        }

      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('user_preference_login_redirect.settings')
      ->set('available_routes', $form_state->getValue('available_routes'))
      ->set('respect_destination_parameter', $form_state->getValue('respect_destination_parameter'))
      ->set('fieldset_title', $form_state->getValue('fieldset_title'))
      ->set('fieldset_weight', $form_state->getValue('fieldset_weight'))
      ->set('field_title', $form_state->getValue('field_title'))
      ->set('field_description', $form_state->getValue('field_description'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
