<?php

declare(strict_types=1);

namespace Drupal\Tests\user_preference_login_redirect\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the User Preference Login Redirect functionality.
 *
 * @group user_preference_login_redirect
 */
class UserPreferenceLoginRedirectTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'test_page_test',
    'user_preference_login_redirect',
  ];

  /**
   * Tests the settings form.
   */
  public function testSettingsForm(): void {
    // Create a user with permission to administer settings.
    $admin_user = $this->drupalCreateUser(['administer user preference login redirect settings']);
    $this->drupalLogin($admin_user);

    // Access the settings form.
    $this->drupalGet('admin/config/people/user-preference-login-redirect');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Available redirect routes');
    $this->assertSession()->pageTextContains('Respect ?destination URL-parameter');
    $this->assertSession()->pageTextContains('Fieldset weight');
    $this->assertSession()->pageTextContains('Fieldset title');
    $this->assertSession()->pageTextContains('Field title');
    $this->assertSession()->pageTextContains('Field description');

    // Verify all fields exist.
    $this->assertSession()->fieldExists('available_routes');
    $this->assertSession()->fieldExists('respect_destination_parameter');
    $this->assertSession()->fieldExists('fieldset_title');
    $this->assertSession()->fieldExists('fieldset_weight');
    $this->assertSession()->fieldExists('field_title');
    $this->assertSession()->fieldExists('field_description');

    // Submit the form with valid data.
    $edit = [
      'available_routes' => "<front>|Default (Frontpage)\nuser.page|My Account",
      'respect_destination_parameter' => TRUE,
      'fieldset_title' => 'Custom Fieldset Title',
      'field_title' => 'Custom Field Title',
      'field_description' => 'Custom field description text',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the configuration was saved correctly.
    $config = $this->config('user_preference_login_redirect.settings');
    $this->assertEquals("<front>|Default (Frontpage)\nuser.page|My Account", $config->get('available_routes'));
    $this->assertEquals(TRUE, $config->get('respect_destination_parameter'));
    $this->assertEquals('Custom Fieldset Title', $config->get('fieldset_title'));
    $this->assertEquals('Custom Field Title', $config->get('field_title'));
    $this->assertEquals('Custom field description text', $config->get('field_description'));
  }

  /**
   * Tests the user profile form field.
   */
  public function testUserProfileField(): void {
    // Configure available routes.
    $config = $this->config('user_preference_login_redirect.settings');
    $config->set('available_routes', "<front>|Default (Frontpage)\nuser.page|My Account");
    $config->save();

    // Create a user with permission to select redirect preference.
    $user = $this->drupalCreateUser(['select login redirect preference in user profile']);
    $this->drupalLogin($user);

    // Access the user edit form.
    $this->drupalGet('user/' . $user->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Login redirect preference');
    $this->assertSession()->fieldExists('redirect_route');

    // Submit the form with a selection.
    $edit = [
      'redirect_route' => 'user.page',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('The changes have been saved.');

    // Verify the preference was saved.
    $user_data = \Drupal::service('user.data');
    $saved_preference = $user_data->get('user_preference_login_redirect', $user->id(), 'redirect_route');
    $this->assertEquals('user.page', $saved_preference);
  }

  /**
   * Tests the login redirect functionality.
   */
  public function testLoginRedirect(): void {
    // Configure available routes.
    $config = $this->config('user_preference_login_redirect.settings');
    $config->set('available_routes', "<front>|Default (Frontpage)\nuser.page|My Account");
    $config->set('respect_destination_parameter', TRUE);
    $config->save();

    // Create a user and set their preference.
    $user = $this->drupalCreateUser(['select login redirect preference in user profile']);
    $user_data = \Drupal::service('user.data');
    $user_data->set('user_preference_login_redirect', $user->id(), 'redirect_route', 'user.page');

    // Log in and verify redirect.
    $this->drupalLogin($user);
    $this->assertSession()->addressEquals('user/' . $user->id());
  }

  // @codingStandardsIgnoreStart
  /**
   * Tests destination parameter respect.
   * @todo This test fails remotely for no apparent reason.
   */
  // public function testDestinationParameterRespect(): void {
  //   // Configure available routes.
  //   $config = $this->config('user_preference_login_redirect.settings');
  //   $config->set('available_routes', "<front>|Default (Frontpage)\nuser.page|My Account");
  //   $config->set('respect_destination_parameter', TRUE);
  //   $config->save();

  //   $user_data = \Drupal::service('user.data');
  //   $user_data->set('user_preference_login_redirect', $this->rootUser->id(), 'redirect_route', 'user.page');

  //   // Log in with a destination parameter.
  //   $this->drupalGet('user/login', ['query' => ['destination' => '/test-page']]);
  //   $edit = [
  //     'name' => $this->rootUser->getAccountName(),
  //     'pass' => $this->rootUser->passRaw,
  //   ];
  //   $this->submitForm($edit, 'Log in');

  //   // The destination parameter should take precedence.
  //   $this->assertSession()->statusCodeEquals(200);
  //   $this->assertSession()->addressEquals('/test-page');
  // }
  // 
  // @codingStandardsIgnoreEnd

  /**
   * Tests custom field labels on user profile form.
   */
  public function testCustomFieldLabels(): void {
    // Configure available routes with custom labels.
    $config = $this->config('user_preference_login_redirect.settings');
    $config->set('available_routes', "<front>|Default (Frontpage)\nuser.page|My Account");
    $config->set('fieldset_title', 'My Custom Fieldset');
    $config->set('field_title', 'My Custom Field');
    $config->set('field_description', 'This is a custom description');
    $config->save();

    // Create a user with permission to select redirect preference.
    $user = $this->drupalCreateUser(['select login redirect preference in user profile']);
    $this->drupalLogin($user);

    // Access the user edit form.
    $this->drupalGet('user/' . $user->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // Verify custom labels are displayed.
    $this->assertSession()->pageTextContains('My Custom Fieldset');
    $this->assertSession()->pageTextContains('My Custom Field');
    $this->assertSession()->pageTextContains('This is a custom description');
  }

  /**
   * Tests default values for configuration fields.
   */
  public function testDefaultConfigurationValues(): void {
    // Create a user with permission to administer settings.
    $admin_user = $this->drupalCreateUser(['administer user preference login redirect settings']);
    $this->drupalLogin($admin_user);

    // Access the settings form.
    $this->drupalGet('admin/config/people/user-preference-login-redirect');
    $this->assertSession()->statusCodeEquals(200);

    // Check that default values are present in the form.
    $this->assertSession()->fieldValueEquals('fieldset_title', 'Login redirect preference');
    $this->assertSession()->fieldValueEquals('field_title', 'Preferred login redirect destination');
    $this->assertSession()->fieldValueEquals('field_description', 'Select where you would like to be redirected after logging in.');
    $this->assertSession()->checkboxChecked('respect_destination_parameter');
  }

  /**
   * Tests settings form validation.
   */
  public function testSettingsFormValidation(): void {
    // Create a user with permission to administer settings.
    $admin_user = $this->drupalCreateUser(['administer user preference login redirect settings']);
    $this->drupalLogin($admin_user);

    // Access the settings form.
    $this->drupalGet('admin/config/people/user-preference-login-redirect');

    // Try to submit with empty available_routes.
    $edit = [
      'available_routes' => '',
      'fieldset_title' => 'Test',
      'field_title' => 'Test',
      'field_description' => 'Test',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Available redirect routes field is required.');

    // Try to submit with invalid route format (missing pipe).
    $edit = [
      'available_routes' => "invalid_route_without_pipe",
      'fieldset_title' => 'Test',
      'field_title' => 'Test',
      'field_description' => 'Test',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Line 1 is invalid');

    // Try to submit with empty fieldset_title.
    $edit = [
      'available_routes' => "<front>|Default",
      'fieldset_title' => '',
      'field_title' => 'Test',
      'field_description' => 'Test',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Fieldset title field is required.');

    // Try to submit with empty field_title.
    $edit = [
      'available_routes' => "<front>|Default",
      'fieldset_title' => 'Test',
      'field_title' => '',
      'field_description' => 'Test',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Field title field is required.');

    // Try to submit with empty field_description.
    $edit = [
      'available_routes' => "<front>|Default",
      'fieldset_title' => 'Test',
      'field_title' => 'Test',
      'field_description' => '',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Field description field is required.');
  }

}
