<?php

declare(strict_types=1);

namespace Drupal\Tests\user_preference_login_redirect\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\user_preference_login_redirect\RouteParser;

/**
 * Tests the route parsing functionality.
 *
 * @group user_preference_login_redirect
 * @coversDefaultClass \Drupal\user_preference_login_redirect\RouteParser
 */
class RouteParserTest extends UnitTestCase {

  /**
   * The RouteParser instance.
   *
   * @var \Drupal\user_preference_login_redirect\RouteParser
   */
  protected $routeParser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->routeParser = new RouteParser();
  }

  /**
   * Tests parsing valid route configuration.
   *
   * @covers ::parseRoutes
   */
  public function testParseValidRoutes(): void {
    $input = "<front>|Default (Frontpage)\nuser.page|My Account\nnode.add|Create Content";

    $result = $this->routeParser->parseRoutes($input);

    $expected = [
      '<front>' => 'Default (Frontpage)',
      'user.page' => 'My Account',
      'node.add' => 'Create Content',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing routes with extra whitespace.
   *
   * @covers ::parseRoutes
   */
  public function testParseRoutesWithWhitespace(): void {
    $input = "  <front>  |  Default (Frontpage)  \n  user.page  |  My Account  ";

    $result = $this->routeParser->parseRoutes($input);

    $expected = [
      '<front>' => 'Default (Frontpage)',
      'user.page' => 'My Account',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing routes with empty lines.
   *
   * @covers ::parseRoutes
   */
  public function testParseRoutesWithEmptyLines(): void {
    $input = "<front>|Default (Frontpage)\n\nuser.page|My Account\n\n";

    $result = $this->routeParser->parseRoutes($input);

    $expected = [
      '<front>' => 'Default (Frontpage)',
      'user.page' => 'My Account',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing invalid routes.
   *
   * @covers ::parseRoutes
   */
  public function testParseInvalidRoutes(): void {
    $input = "invalid_route_without_pipe\n<front>|Valid Route";

    $result = $this->routeParser->parseRoutes($input);

    // Only the valid route should be parsed.
    $expected = [
      '<front>' => 'Valid Route',
    ];

    $this->assertEquals($expected, $result);
  }

  /**
   * Tests parsing empty string.
   *
   * @covers ::parseRoutes
   */
  public function testParseEmptyString(): void {
    $input = '';

    $result = $this->routeParser->parseRoutes($input);

    $this->assertEmpty($result);
  }

}
