<?php

namespace Drupal\user_provisioning\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Drupal\user_provisioning\moUserProvisioningSupport;
use Drupal\user_provisioning\moUserProvisioningUtilities;

/**
 *
 */
class MoUserProvisioningRequestTrial extends FormBase {
  private ImmutableConfig $config;
  protected $messenger;

  /**
   *
   */
  public function __construct() {
    $this->messenger = \Drupal::messenger();
  }

  /**
   *
   */
  public function getFormId() {
    return 'user_provisioning_request_trial';
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state, $options = NULL) {

    $form['#prefix'] = '<div id="modal_example_form">';
    $form['#suffix'] = '</div>';
    $form['status_messages'] = [
      '#type' => 'status_messages',
      '#weight' => -10,
    ];

    $form['radio_option'] = [
      '#type' => 'radios',
      '#title' => $this->t('Which type of trial would you prefer?'),
      '#options' => [
        'option1' => $this->t('Sandbox'),
        'option2' => $this->t('On-Premise'),
      ],
      '#default_value' => ($form_state->getValue('radio_option')) ? $form_state->getValue('radio_option') : 'option1',
      '#attributes' => array('class' => array('container-inline'),),
      '#ajax' => [
        'callback' => '::updateFormElements',
        'wrapper' => 'user-prov-additional-fields-wrapper',
      ],
    ];

    $form['user_prov_additional_fields_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'user-prov-additional-fields-wrapper'],
    ];


    $form['mo_user_provisioning_trial_email_address'] = [
      '#type' => 'email',
      '#title' => t('Email'),
      '#default_value' => self::getEmail(),
      '#required' => TRUE,
      '#attributes' => ['placeholder' => t('Enter your email'), 'style' => 'width:99%;margin-bottom:1%;'],
      '#states' => [
        'visible' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],
        ],
        'required' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],],
      ],
    ];

    $form['mo_user_provisioning_trial_method'] = [
      '#type' => 'select',
      '#title' => t('Trial Method'),
      '#attributes' => ['style' => 'width:99%;height:30px;margin-bottom:1%;'],
      '#options' => [
        'Drupal ' . moUserProvisioningUtilities::mo_get_drupal_core_version() . ' SCIM Client' => t('Drupal ' . moUserProvisioningUtilities::mo_get_drupal_core_version() . ' SCIM CLIENT - Sync information from Drupal'),
        'Drupal ' . moUserProvisioningUtilities::mo_get_drupal_core_version() . ' SCIM SERVER' => t('Drupal ' . moUserProvisioningUtilities::mo_get_drupal_core_version() . ' SCIM SERVER - Sync information into Drupal'),
        'Drupal ' . moUserProvisioningUtilities::mo_get_drupal_core_version() . ' PROVIDER SPECIFIC PROVISIONING' => t('Drupal ' . moUserProvisioningUtilities::mo_get_drupal_core_version() . ' Two-way sync between Drupal and provider'),
        'Not Sure' => t('Not Sure (Want to schedule a meeting with Drupal developer for assistance.)'),
      ],
      '#states' => [
        'visible' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],
        ],
        'required' => array(
          ':input[name="radio_option"]' => ['value' => 'option2'],),
      ],
    ];

    $form['mo_user_provisioning_trial_provider'] = [
      '#type' => 'textarea',
      '#rows' => 1,
      '#title' => t('Provider/Application'),
      '#attributes' => ['placeholder' => t('Enter provider name with which you want to sync.'), 'style' => 'width:99%;'],
      '#states' => [
        'visible' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],
        ],
        'required' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],],
      ],
    ];

    $form['mo_user_provisioning_trial_description'] = [
      '#type' => 'textarea',
      '#rows' => 3,
      '#title' => t('Description'),
      '#attributes' => ['placeholder' => t('Describe your use case here!'), 'style' => 'width:99%;'],
      '#states' => [
        'visible' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],
        ],
        'required' => array(
          ':input[name="radio_option"]' => ['value' => 'option2'],),
      ],
    ];

    $form['mo_user_prov_trial_submit_button_option1'] = [
      '#type' => 'submit',
      '#value' => $this->t('Go to Sandbox'),
      '#attributes' => [
        'class' => ['option1-submit','use-ajax', 'button--primary'],
        'formtarget' => '_blank'
      ],
      '#prefix' => '<div class="option1-submit-wrapper">',
      '#suffix' => '</div>',
      '#states' => [
        'visible' => [
          ':input[name="radio_option"]' => ['value' => 'option1'],],
      ],
      '#submit' => ['::goToSandbox',],
    ];

    $form['mo_user_prov_trial_submit_button_option'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
      '#attributes' => [
        'class' => [
          'use-ajax',
          'button--primary',
        ],
      ],
      '#prefix' => '<div class="other-options-submit-wrapper">',
      '#suffix' => '</div>',
      '#states' => [
        'visible' => [
          ':input[name="radio_option"]' => ['value' => 'option2'],
        ],
      ],
      '#ajax' => [
        'callback' => [$this, 'submitModalFormAjax'],
        'event' => 'click',
      ],
    ];

    $form['#attached']['library'][] = 'core/drupal.dialog.ajax';
    return $form;
  }

  /**
   * @param array $form
   * @param FormStateInterface $form_state
   * @return mixed
   */
  public function updateFormElements(array &$form, FormStateInterface $form_state) {
    return $form['user_prov_additional_fields_wrapper'];
  }

  /**
   *
   */
  public function submitModalFormAjax(array $form, FormStateInterface $form_state) {
    $form_values = $form_state->getValues();
    $response = new AjaxResponse();
    // If there are any form errors, AJAX replace the form.
    if ($form_state->hasAnyErrors()) {
      $response->addCommand(new ReplaceCommand('#modal_example_form', $form));
    }
    else {

      $email = $form_values['mo_user_provisioning_trial_email_address'];
      if(!filter_var($email, FILTER_VALIDATE_EMAIL)){
        \Drupal::messenger()->addMessage(t('The email address <b><i>' . $email . '</i></b> is not valid.'), 'error');
        $response->addCommand(new ReplaceCommand('#modal_example_form', $form));
        return $response;
      }
      $requested_trial_version = $form_values['mo_user_provisioning_trial_method'];
      $provider = $form_values['mo_user_provisioning_trial_provider'];
      $query = $requested_trial_version . ' : ' . trim($form_values['mo_user_provisioning_trial_description']);
      $query_type = 'Trial Request';

      $support = new moUserProvisioningSupport($email, '', $query, $query_type, $requested_trial_version, $provider);
      $support_response = $support->sendSupportQuery();

      $this->messenger->addStatus(t('Success! Trial query successfully sent. We will provide you with the trial version shortly.'));
      $response->addCommand(new RedirectCommand(Url::fromRoute('user_provisioning.provisioning')->toString()));
    }
    return $response;
  }

  /**
   * Function to redirect user to Sandbox
   *
   * @param array $form
   * @param FormStateInterface $form_state
   * @return void
   */
  public function goToSandbox(array $form, FormStateInterface $form_state) {
    $url = Url::fromUri('https://playground.miniorange.com/drupal.php',[
      'query' => [
        'email' => self::getEmail(),
        'mo_module' => 'user_provisioning',
        'drupal_version' => '10',
      ],
    ])->toString();
    $response = new TrustedRedirectResponse($url);
    $form_state->setResponse($response);
  }

  /**
   * Function to fetch user email.
   *
   * @return array|mixed|string|null
   */
  public static function getEmail(){
    $user  = \Drupal::currentUser()->getEmail();
    $email = \Drupal::config('user_provisioning.settings')->get('mo_user_provisioning_customer_email');
    $email = !empty($email) ? $email : $user;
    $email = preg_match('/^(?!.*(?:noreply|no-reply)).*$/i', $email) ? $email : '';
    return $email;
  }

  /**
   *
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
  }

}
