<?php

namespace Drupal\user_reference_invite\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for User Invite Reference settings.
 */
class InviteSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['user_reference_invite.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_reference_invite_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('user_reference_invite.settings');

    $form['expiry'] = [
      '#type' => 'details',
      '#title' => $this->t('Invitation Expiry'),
      '#open' => TRUE,
    ];

    $form['expiry']['default_expiry_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Default expiry (days)'),
      '#default_value' => $config->get('default_expiry_days') ?? 30,
      '#min' => 1,
      '#max' => 365,
    ];

    $form['email'] = [
      '#type' => 'details',
      '#title' => $this->t('Email Settings'),
      '#open' => TRUE,
    ];

    $form['email']['send_invitation_email'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send invitation emails'),
      '#default_value' => $config->get('send_invitation_email') ?? TRUE,
    ];

    // Role assignment settings.
    $form['role_assignment'] = [
      '#type' => 'details',
      '#title' => $this->t('Role Assignment'),
      '#open' => TRUE,
    ];

    $form['role_assignment']['allow_role_assignment_on_content'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow "Assign roles" on content'),
      '#description' => $this->t('When enabled, content editors can select which roles to assign to invited users. When disabled, the default role will be assigned automatically.'),
      '#default_value' => $config->get('allow_role_assignment_on_content'),
    ];

    // Get all user roles except anonymous.
    $role_storage = \Drupal::entityTypeManager()->getStorage('user_role');
    $roles = [];
    foreach ($role_storage->loadMultiple() as $role_id => $role) {
      if ($role_id !== 'anonymous') {
        $roles[$role_id] = $role->label();
      }
    }

    $form['role_assignment']['default_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Default roles'),
      '#description' => $this->t('These roles will be automatically assigned to invited users when role selection is disabled.'),
      '#options' => $roles,
      '#default_value' => $config->get('default_roles') ?? [],
      '#states' => [
        'visible' => [
          ':input[name="allow_role_assignment_on_content"]' => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $form['role_assignment']['allowed_global_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed roles for assignment'),
      '#description' => $this->t('Select which roles can be assigned to invited users. Leave empty to allow all roles.'),
      '#options' => $roles,
      '#default_value' => $config->get('allowed_global_roles') ?? [],
      '#states' => [
        'visible' => [
          ':input[name="allow_role_assignment_on_content"]' => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Filter out unchecked roles.
    $allowed_roles = array_filter($form_state->getValue('allowed_global_roles'));
    $default_roles = array_filter($form_state->getValue('default_roles'));

    $this->config('user_reference_invite.settings')
      ->set('default_expiry_days', $form_state->getValue('default_expiry_days'))
      ->set('send_invitation_email', $form_state->getValue('send_invitation_email'))
      ->set('allow_role_assignment_on_content', $form_state->getValue('allow_role_assignment_on_content'))
      ->set('default_roles', array_values($default_roles))
      ->set('allowed_global_roles', array_values($allowed_roles))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
