# User Reference with Invite

A Drupal module that enhances User Reference fields with invitation workflow capabilities.

## Features

- **Dual Selection Mode**: Select existing users OR invite new users via email
- **Automatic Field Attachment**: Users automatically added to reference field after registration
- **Secure Invitations**: Cryptographic tokens with expiration
- **Context-Specific Roles**: Assign roles based on invitation context
- **Admin Interface**: Manage all invitations at `/admin/people/invitations`
- **Email Templates**: Fully customizable invitation emails via Drupal's user account settings
- **Token Support**: Rich token system for dynamic email content
- **HTML Mail Support**: Automatic HTML mail support when htmlmail/swiftmailer modules are enabled
- **Entity Agnostic**: Works with any entity type and user reference field

## Installation

1. Place this module in `web/modules/custom/user_reference_invite`
2. Enable the module: `drush en user_reference_invite`
3. Configure permissions at `/admin/people/permissions`
4. Configure settings at `/admin/config/people/user-reference-invite`

## Usage

### For Site Builders

1. Go to the field settings for any User Reference field
2. Select "User Reference with Invite" as the widget
3. Configure widget settings (invitation expiry, allowed roles, etc.)
4. Save the field configuration

### For Content Editors

When editing an entity with the enhanced field:

1. Choose between "Select existing user" or "Invite new users by email"
2. For existing users: Use autocomplete to select users (supports multiple selections)
3. For new users:
   - Enter one or more email addresses separated by commas
   - Example: `user1@example.com, user2@example.com, user3@example.com`
   - Optionally assign roles that will be applied to all invited users
   - The "Add another item" button is hidden in invite mode for a cleaner interface
4. Invited users receive an email with registration/login link
5. After registration, users are automatically added to the field

### For Administrators

View and manage all invitations at `/admin/people/invitations`:
- Filter by status (Pending, Accepted, Expired)
- Filter by entity type
- Cancel pending invitations
- View invitation history

## Configuration

### Email Templates

**`/admin/config/people/accounts`**

The module integrates with Drupal's standard user account email settings. Under the "User Invitation" section, you can configure:

- **Subject**: Email subject line with token support
- **Body**: Email body content with full HTML and token support

**Available Tokens**:
- `[user_invite:invited_by]` - Name of the user who sent the invitation
- `[user_invite:entity_label]` - Label of the entity being invited to
- `[user_invite:register_url]` - Registration URL with invitation token
- `[user_invite:login_url]` - Login URL for existing users
- `[user_invite:accept_url]` - Direct acceptance URL
- `[user_invite:expiry_date]` - When the invitation expires
- `[user_invite:email]` - Invited user's email address
- `[user_invite:roles]` - Roles to be assigned
- `[site:name]` - Site name
- `[site:url]` - Site URL

The Token module (if installed) provides a token browser for easy token insertion.

### Global Settings

**`/admin/config/people/user-invite-reference`**

- **Expiry**: Default and maximum invitation expiry days
- **Rate Limiting**: Limit invitations per user/email
- **Auto-Attachment**: Configure automatic field attachment
- **Cleanup**: Configure expired invitation cleanup

### Field Widget Settings

When configuring a field:
- Enable/disable invitations
- Set default expiry for this field
- Choose allowed roles for invited users

## Permissions

- **Administer user invitations**: Full access to all invitations
- **Invite users via reference fields**: Create invitations
- **View own invitations**: View sent invitations
- **Cancel own invitations**: Cancel pending invitations

## Architecture

### Core Components

1. **UserInvite Entity**: Tracks invitation state
2. **TokenService**: Generates secure invitation tokens
3. **UserInviteManager**: Business logic for invitations
4. **UserReferenceInviteWidget**: Custom field widget
5. **Event Subscribers**: Handle user registration and cleanup

### Database Schema

**user_invite table**:
- id, uuid, langcode
- email, token, status
- entity_type, entity_id, field_name
- roles (JSON)
- invited_by, created, expires, accepted
- metadata (JSON)

### Workflow

1. User creates invitation via widget
2. System generates secure token
3. Email sent to invitee
4. Invitee clicks link and registers/logs in
5. Event subscriber detects registration
6. User automatically attached to reference field
7. Context-specific roles assigned

## Security

- SHA-256 hashed tokens (not stored in plain text)
- Time-limited invitations (default 30 days)
- Rate limiting to prevent abuse
- Permission-based access control
- Input validation (email format, entity existence)
- CSRF protection via Form API

## API

### Programmatic Usage

```php
// Get the invite manager service
$inviteManager = \Drupal::service('user_reference_invite.invite_manager');

// Create an invitation
$invite = $inviteManager->createInvitation(
  'user@example.com',
  'node',
  123,
  'field_team_members',
  ['editor'], // Roles
  ['custom_data' => 'value'] // Metadata
);

// Send invitation email
$inviteManager->sendInvitationEmail($invite);

// Find pending invitations
$pending = $inviteManager->findPendingInvitationsByEmail('user@example.com');

// Accept invitation
$inviteManager->acceptInvitation($token, $user);
```

### Custom Tokens

The module provides a custom `user_invite` token type. You can use these tokens in:
- Email templates at `/admin/config/people/accounts`
- Custom modules via `\Drupal::token()->replace()`
- Any Drupal email or text that supports token replacement

Example usage in custom code:
```php
$token_data = ['user_invite' => $params];
$message = \Drupal::token()->replace('[user_invite:invited_by] invited you to [user_invite:entity_label]', $token_data);
```

### Email Template Integration

The module uses Drupal's standard email template system:

1. Templates are stored in `user.mail` configuration
2. Tokens are replaced during email sending
3. HTML support is automatic when HTML mail modules are enabled
4. Admins can customize templates without code changes

### Hooks

The module integrates with standard Drupal hooks:
- `hook_mail()` - Defines the invitation email
- `hook_mail_alter()` - Processes email templates and tokens
- `hook_token_info()` - Defines available tokens
- `hook_tokens()` - Replaces token values
- `hook_form_FORM_ID_alter()` - Adds settings to user account config

## Troubleshooting

### Invitations not sending
- Check email configuration
- Verify `send_invitation_email` setting is enabled
- Check Drupal logs

### Users not automatically attached
- Verify `auto_attach_on_registration` is enabled
- Check that parent entity still exists
- Verify field exists on entity

### Token errors
- Tokens expire after configured days
- Ensure token in URL matches database
- Check for URL encoding issues

## Maintenance

### Cron Tasks

The module automatically cleans up:
- Expired invitations (after configured days)
- Accepted invitations (after retention period)

Run cron regularly: `drush cron`

### Manual Cleanup

```bash
# Clean up expired invitations
drush eval "\Drupal::service('user_reference_invite.invite_manager')->cleanupExpiredInvitations();"
```

## Upgrading from Previous Versions

### Version 2.x (Email Template Migration)

If you're upgrading from a version that used custom message fields in the widget or config:

1. Run database updates: `drush updb`
2. The update hook will automatically:
   - Remove deprecated `custom_message_enabled` and `custom_message_default` settings
   - Initialize default email templates in Drupal's user.mail configuration
3. Visit `/admin/config/people/accounts` to customize your invitation email templates
4. Old custom messages are not automatically migrated - you'll need to manually add any custom content to the new templates

## Future Enhancements

- Bulk invitation upload (CSV)
- Invitation analytics dashboard
- REST API endpoints
- Integration with social login
- Custom approval workflows
- Multi-language email templates

## Support

For issues, feature requests, or contributions, please contact the development team.

## License

This module is licensed under the same terms as Drupal core.
