<?php

namespace Drupal\user_reference_invite\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Access controller for the User Invite entity.
 */
class InviteAccessController extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\user_reference_invite\Entity\UserInviteInterface $entity */
    switch ($operation) {
      case 'view':
        // Admin can view all.
        if ($account->hasPermission('administer user invitations')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        // Users can view their own.
        if ($account->hasPermission('view own invitations')
            && $entity->getInvitedBy() == $account->id()) {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }
        return AccessResult::forbidden();

      case 'update':
        // Only admin can update.
        return AccessResult::allowedIfHasPermission($account, 'administer user invitations');

      case 'delete':
        // Admin can delete all.
        if ($account->hasPermission('administer user invitations')) {
          return AccessResult::allowed()->cachePerPermissions();
        }
        // Users can delete their own pending invitations.
        if ($account->hasPermission('cancel own invitations')
            && $entity->getInvitedBy() == $account->id()
            && $entity->getStatus() === 'pending') {
          return AccessResult::allowed()
            ->cachePerPermissions()
            ->cachePerUser()
            ->addCacheableDependency($entity);
        }
        return AccessResult::forbidden();

      default:
        return AccessResult::neutral();
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    return AccessResult::allowedIfHasPermission($account, 'invite users via reference fields');
  }

}
