<?php

namespace Drupal\user_reference_invite\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\user_reference_invite\Service\UserInviteManagerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Route;

/**
 * Determines access to user registration route.
 */
class UserRegisterAccess implements AccessInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The invite manager.
   *
   * @var \Drupal\user_reference_invite\Service\UserInviteManagerInterface
   */
  protected $inviteManager;

  /**
   * Constructs a UserRegisterAccess object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\user_reference_invite\Service\UserInviteManagerInterface $invite_manager
   *   The invite manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RequestStack $request_stack,
    UserInviteManagerInterface $invite_manager,
  ) {
    $this->configFactory = $config_factory;
    $this->requestStack = $request_stack;
    $this->inviteManager = $invite_manager;
  }

  /**
   * Checks access to user registration.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The currently logged in account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, AccountInterface $account) {
    // Logged in users can't register.
    if ($account->isAuthenticated()) {
      return AccessResult::forbidden()->cachePerUser();
    }

    // Check for invite token in the request.
    $request = $this->requestStack->getCurrentRequest();
    $token = $request ? $request->query->get('invite_token') : NULL;

    // If there's a valid invite token, allow access.
    if ($token && $this->inviteManager->validateToken($token)) {
      return AccessResult::allowed()
        ->cachePerUser()
        ->addCacheContexts(['url.query_args:invite_token']);
    }

    // Otherwise, use the standard registration access check.
    $config = $this->configFactory->get('user.settings');
    $register_setting = $config->get('register');

    // Visitors can register themselves, no administrator approval required.
    if ($register_setting === 'visitors') {
      return AccessResult::allowed()
        ->cachePerUser()
        ->addCacheableDependency($config);
    }

    // Visitors can register, but administrator approval is required.
    if ($register_setting === 'visitors_admin_approval') {
      return AccessResult::allowed()
        ->cachePerUser()
        ->addCacheableDependency($config);
    }

    // Only administrators can create accounts; deny access.
    return AccessResult::forbidden()
      ->cachePerUser()
      ->addCacheableDependency($config);
  }

}
