<?php

namespace Drupal\user_reference_invite\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the User Invite entity.
 *
 * @ContentEntityType(
 *   id = "user_invite",
 *   label = @Translation("User Invitation"),
 *   base_table = "user_invite",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "langcode" = "langcode",
 *   },
 *   handlers = {
 *     "access" = "Drupal\user_reference_invite\Access\InviteAccessController",
 *     "list_builder" = "Drupal\Core\Entity\EntityListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *   },
 *   admin_permission = "administer user invitations",
 * )
 */
class UserInvite extends ContentEntityBase implements UserInviteInterface {

  /**
   * {@inheritdoc}
   */
  public function getEmail() {
    return $this->get('email')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setEmail($email) {
    $this->set('email', $email);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getToken() {
    return $this->get('token')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setToken($token) {
    $this->set('token', $token);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus() {
    return $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setStatus($status) {
    $this->set('status', $status);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isExpired() {
    $expires = $this->getExpiresTime();
    return $expires && $expires < \Drupal::time()->getRequestTime();
  }

  /**
   * {@inheritdoc}
   */
  public function isPending() {
    return $this->getStatus() === 'pending' && !$this->isExpired();
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntityType() {
    return $this->get('entity_type')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetEntityId() {
    return $this->get('entity_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldName() {
    return $this->get('field_name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getInvitedBy() {
    return $this->get('invited_by')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getExpiresTime() {
    return $this->get('expires')->value;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['email'] = BaseFieldDefinition::create('email')
      ->setLabel(t('Email'))
      ->setDescription(t('The email address of the invited user.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 254,
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['token'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Token'))
      ->setDescription(t('The unique invitation token.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 64,
      ]);

    $fields['status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Status'))
      ->setDescription(t('The invitation status.'))
      ->setRequired(TRUE)
      ->setDefaultValue('pending')
      ->setSettings([
        'allowed_values' => [
          'pending' => 'Pending',
          'accepted' => 'Accepted',
          'expired' => 'Expired',
          'cancelled' => 'Cancelled',
        ],
      ])
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'list_default',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['entity_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Entity Type'))
      ->setDescription(t('The entity type this invitation is for.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 32,
      ]);

    $fields['entity_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Entity ID'))
      ->setDescription(t('The entity ID this invitation is for.'))
      ->setRequired(TRUE);

    $fields['field_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Field Name'))
      ->setDescription(t('The field name this invitation is for.'))
      ->setRequired(TRUE)
      ->setSettings([
        'max_length' => 32,
      ]);

    $fields['roles'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Roles'))
      ->setDescription(t('JSON-encoded roles to assign.'))
      ->setDefaultValue('[]');

    $fields['invited_by'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Invited By'))
      ->setDescription(t('The user who created this invitation.'))
      ->setRequired(TRUE)
      ->setSetting('target_type', 'user')
      ->setDisplayOptions('view', [
        'label' => 'inline',
        'type' => 'entity_reference_label',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time the invitation was created.'));

    $fields['expires'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Expires'))
      ->setDescription(t('The time the invitation expires.'))
      ->setRequired(TRUE);

    $fields['accepted'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Accepted'))
      ->setDescription(t('The time the invitation was accepted.'));

    $fields['accepted_by'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Accepted By'))
      ->setDescription(t('The user who accepted this invitation.'))
      ->setSetting('target_type', 'user');

    $fields['metadata'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Metadata'))
      ->setDescription(t('JSON-encoded additional data.'))
      ->setDefaultValue('{}');

    return $fields;
  }

}
