<?php

namespace Drupal\user_reference_invite\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\user_reference_invite\Service\UserInviteManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to allow registration when valid invite token is present.
 */
class RegistrationAccessSubscriber implements EventSubscriberInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The invite manager.
   *
   * @var \Drupal\user_reference_invite\Service\UserInviteManagerInterface
   */
  protected $inviteManager;

  /**
   * Constructs a new RegistrationAccessSubscriber.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\user_reference_invite\Service\UserInviteManagerInterface $invite_manager
   *   The invite manager service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RouteMatchInterface $route_match,
    UserInviteManagerInterface $invite_manager,
  ) {
    $this->configFactory = $config_factory;
    $this->routeMatch = $route_match;
    $this->inviteManager = $invite_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Run early to allow registration access before other checks.
    $events[KernelEvents::REQUEST][] = ['onRequest', 100];
    return $events;
  }

  /**
   * Temporarily allow user registration when valid invite token is present.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onRequest(RequestEvent $event) {
    $request = $event->getRequest();
    $route_name = $this->routeMatch->getRouteName();

    // Only act on user registration route.
    if ($route_name !== 'user.register') {
      return;
    }

    // Check if invite token is present.
    $token = $request->query->get('invite_token');
    if (!$token) {
      return;
    }

    // Validate the token.
    $invite = $this->inviteManager->validateToken($token);
    if (!$invite) {
      return;
    }

    // By Passing the registration setting to allow this user to register.
    $request->attributes->set('_user_invite_bypass_registration', TRUE);
  }

}
