<?php

namespace Drupal\user_reference_invite\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for User Invite Reference settings.
 */
class InviteSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['user_reference_invite.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_reference_invite_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('user_reference_invite.settings');

    $form['expiry'] = [
      '#type' => 'details',
      '#title' => $this->t('Invitation Expiry'),
      '#open' => TRUE,
    ];

    $form['expiry']['default_expiry_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Default expiry (days)'),
      '#default_value' => $config->get('default_expiry_days') ?? 30,
      '#min' => 1,
      '#max' => 365,
    ];

    $form['email'] = [
      '#type' => 'details',
      '#title' => $this->t('Email Settings'),
      '#open' => TRUE,
    ];

    $form['email']['send_invitation_email'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send invitation emails'),
      '#default_value' => $config->get('send_invitation_email') ?? TRUE,
    ];

    // Role assignment settings.
    $form['role_assignment'] = [
      '#type' => 'details',
      '#title' => $this->t('Role Assignment'),
      '#open' => TRUE,
    ];

    $form['role_assignment']['allow_role_assignment_on_content'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow "Assign roles" on content'),
      '#description' => $this->t('When enabled, content editors can select which roles to assign to invited users. When disabled, the default role will be assigned automatically.'),
      '#default_value' => $config->get('allow_role_assignment_on_content'),
    ];

    // Get all user roles except anonymous.
    $role_storage = \Drupal::entityTypeManager()->getStorage('user_role');
    $roles = [];
    foreach ($role_storage->loadMultiple() as $role_id => $role) {
      if ($role_id !== 'anonymous') {
        $roles[$role_id] = $role->label();
      }
    }

    $form['role_assignment']['default_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Default roles'),
      '#description' => $this->t('These roles will be automatically assigned to invited users when role selection is disabled.'),
      '#options' => $roles,
      '#default_value' => $config->get('default_roles') ?? [],
      '#states' => [
        'visible' => [
          ':input[name="allow_role_assignment_on_content"]' => [
            'checked' => FALSE,
          ],
        ],
      ],
    ];

    $form['role_assignment']['allowed_global_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed roles for assignment'),
      '#description' => $this->t('Select which roles can be assigned to invited users. Leave empty to allow all roles.'),
      '#options' => $roles,
      '#default_value' => $config->get('allowed_global_roles') ?? [],
      '#states' => [
        'visible' => [
          ':input[name="allow_role_assignment_on_content"]' => [
            'checked' => TRUE,
          ],
        ],
      ],
    ];

    // Table display settings.
    $form['table_display'] = [
      '#type' => 'details',
      '#title' => $this->t('Table Display Settings'),
      '#description' => $this->t('Configure which columns to display in various tables.'),
      '#open' => TRUE,
    ];

    // Admin page table configuration.
    $admin_columns = $this->getAvailableAdminColumns();
    $default_admin_columns = $config->get('admin_table_columns');
    if (empty($default_admin_columns)) {
      $default_admin_columns = array_keys($admin_columns);
    }

    $form['table_display']['admin_table_columns'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Admin Page - Visible columns'),
      '#description' => $this->t('Configure columns for /admin/people/invitations'),
      '#options' => $admin_columns,
      '#default_value' => $default_admin_columns,
      '#required' => TRUE,
    ];

    // Current Users table configuration.
    $current_users_columns = $this->getAvailableCurrentUsersColumns();
    $default_current_users_columns = $config->get('current_users_table_columns');
    if (empty($default_current_users_columns)) {
      $default_current_users_columns = ['name', 'email', 'roles'];
    }

    $form['table_display']['current_users_table_columns'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Current Users Table - Visible columns'),
      '#description' => $this->t('Configure columns for the "Current Users" table in field widget'),
      '#options' => $current_users_columns,
      '#default_value' => $default_current_users_columns,
      '#required' => TRUE,
    ];

    // Pending Invitations table configuration.
    $pending_columns = $this->getAvailablePendingInvitationsColumns();
    $default_pending_columns = $config->get('pending_invitations_table_columns');
    if (empty($default_pending_columns)) {
      $default_pending_columns = ['email', 'sent', 'expires'];
    }

    $form['table_display']['pending_invitations_table_columns'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Pending Invitations Table - Visible columns'),
      '#description' => $this->t('Configure columns for the "Pending Invitations" table in field widget'),
      '#options' => $pending_columns,
      '#default_value' => $default_pending_columns,
      '#required' => TRUE,
    ];

    $form['table_display']['column_order_info'] = [
      '#markup' => '<p><em>' . $this->t('Note: Columns will be displayed in the order selected above.') . '</em></p>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Filter out unchecked roles.
    $allowed_roles = array_filter(
          $form_state->getValue('allowed_global_roles')
      );
    $default_roles = array_filter($form_state->getValue('default_roles'));

    // Filter out unchecked columns for each table.
    $admin_table_columns = array_filter(
          $form_state->getValue('admin_table_columns')
      );
    $current_users_columns = array_filter(
          $form_state->getValue('current_users_table_columns')
      );
    $pending_invitations_columns = array_filter(
          $form_state->getValue('pending_invitations_table_columns')
      );

    $this->config('user_reference_invite.settings')
      ->set('default_expiry_days', $form_state->getValue('default_expiry_days'))
      ->set('send_invitation_email', $form_state->getValue('send_invitation_email'))
      ->set('allow_role_assignment_on_content', $form_state->getValue('allow_role_assignment_on_content'))
      ->set('default_roles', array_values($default_roles))
      ->set('allowed_global_roles', array_values($allowed_roles))
      ->set('admin_table_columns', array_values($admin_table_columns))
      ->set('current_users_table_columns', array_values($current_users_columns))
      ->set('pending_invitations_table_columns', array_values($pending_invitations_columns))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available columns for the admin table.
   *
   * @return array
   *   Array of column machine names and labels.
   */
  protected function getAvailableAdminColumns() {
    return [
      'email' => $this->t('Email'),
      'status' => $this->t('Status'),
      'entity' => $this->t('Entity'),
      'field' => $this->t('Field'),
      'invited_by' => $this->t('Invited By'),
      'created' => $this->t('Created'),
      'expires' => $this->t('Expires'),
      'accepted' => $this->t('Accepted Date'),
      'accepted_by' => $this->t('Accepted By'),
      'roles' => $this->t('Roles'),
    ];
  }

  /**
   * Get available columns for the current users table.
   *
   * @return array
   *   Array of column machine names and labels.
   */
  protected function getAvailableCurrentUsersColumns() {
    return [
      'name' => $this->t('Name'),
      'email' => $this->t('Email'),
      'roles' => $this->t('Roles'),
      'status' => $this->t('Status'),
      'last_access' => $this->t('Last Access'),
    ];
  }

  /**
   * Get available columns for the pending invitations table.
   *
   * @return array
   *   Array of column machine names and labels.
   */
  protected function getAvailablePendingInvitationsColumns() {
    return [
      'email' => $this->t('Email'),
      'sent' => $this->t('Sent'),
      'expires' => $this->t('Expires'),
      'invited_by' => $this->t('Invited By'),
      'roles' => $this->t('Roles'),
    ];
  }

}
