<?php

namespace Drupal\user_reference_invite\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\EntityReferenceAutocompleteWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Utility\Html;

/**
 * Plugin implementation of the 'user_reference_invite' widget.
 *
 * @FieldWidget(
 *   id = "user_reference_invite",
 *   label = @Translation("User Reference with Invite"),
 *   description = @Translation("Autocomplete user reference with invitation capability"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class UserReferenceInviteWidget extends EntityReferenceAutocompleteWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'enable_invitations' => TRUE,
      'show_pending_invites' => TRUE,
      'default_expiry_days' => 30,
      'allowed_roles' => [],
      'selection_modes' => 'both',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    // Check field cardinality and show warning if limited to 1.
    $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    if ($cardinality == 1) {
      $elements['cardinality_warning'] = [
        '#markup' => '<div class="messages messages--warning">' .
        $this->t('Warning: This field is limited to 1 value. The "User Reference with Invite" widget works best with unlimited or multiple values. Consider changing the field cardinality to unlimited or a value greater than 1.') .
        '</div>',
        '#weight' => -100,
      ];
    }

    // Note: Invitations are always enabled when this widget is selected.
    // No need for a separate "Enable invitations" checkbox.
    $elements['show_pending_invites'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show pending invitations'),
      '#description' => $this->t('Display a list of pending invitations for this field.'),
      '#default_value' => $this->getSetting('show_pending_invites'),
    ];

    $elements['default_expiry_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Default invitation expiry (days)'),
      '#description' => $this->t('Number of days before invitation expires.'),
      '#default_value' => $this->getSetting('default_expiry_days'),
      '#min' => 1,
      '#max' => 365,
    ];

    // Get available roles.
    // Load roles excluding anonymous and authenticated.
    $role_storage = \Drupal::entityTypeManager()->getStorage('user_role');
    $roles = [];
    foreach ($role_storage->loadMultiple() as $role_id => $role) {
      if ($role_id !== 'anonymous' && $role_id !== 'authenticated') {
        $roles[$role_id] = $role->label();
      }
    }

    $elements['allowed_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed roles for invitations'),
      '#description' => $this->t('Roles that can be assigned to invited users. Leave empty to not assign any roles.'),
      '#options' => $roles,
      '#default_value' => $this->getSetting('allowed_roles'),
    ];

    $elements['selection_modes'] = [
      '#type' => 'radios',
      '#title' => $this->t('Available selection modes'),
      '#description' => $this->t('Choose which modes to display to users.'),
      '#options' => [
        'both' => $this->t('Both - Select existing and Invite new users'),
        'invite_only' => $this->t('Invite only - Only show invite option'),
      ],
      '#default_value' => $this->getSetting('selection_modes'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    if ($this->getSetting('enable_invitations')) {
      $summary[] = $this->t('Invitations: Enabled');
      $summary[] = $this->t(
            'Expiry: @days days', [
              '@days' => $this->getSetting('default_expiry_days'),
            ]
        );

      $allowed_roles = array_filter($this->getSetting('allowed_roles'));
      if (!empty($allowed_roles)) {
        $summary[] = $this->t(
          'Allowed roles: @count', [
            '@count' => count($allowed_roles),
          ]
          );
      }

      // Add selection modes summary.
      $selection_modes = $this->getSetting('selection_modes');
      if ($selection_modes === 'invite_only') {
        $summary[] = $this->t('Mode: Invite only');
      }
      else {
        $summary[] = $this->t('Mode: Both (existing + invite)');
      }
    }
    else {
      $summary[] = $this->t('Invitations: Disabled');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state) {
    $elements = parent::formMultipleElements($items, $form, $form_state);

    // Only add extra sections if invitations are enabled.
    if (!$this->getSetting('enable_invitations')) {
      return $elements;
    }

    // Get selection modes setting.
    $selection_modes = $this->getSetting('selection_modes');

    // If in "invite_only" mode, hide the default widget elements.
    if ($selection_modes === 'invite_only') {
      // Disable table drag-drop formatting by removing the theme wrapper.
      // This prevents the drag handles and table structure from appearing.
      if (isset($elements['#theme'])) {
        unset($elements['#theme']);
      }
      if (isset($elements['#type']) && $elements['#type'] === 'table') {
        $elements['#type'] = 'container';
      }

      // Hide the "Add another item" button.
      if (isset($elements['add_more'])) {
        $elements['add_more']['#access'] = FALSE;
      }

      // Hide all field items except delta 0 (which has the email field).
      foreach ($elements as $key => $element) {
        if (is_numeric($key) && $key > 0) {
          $elements[$key]['#access'] = FALSE;
        }
      }

      // Hide drag-drop weight column for delta 0 if it exists.
      if (isset($elements[0]['_weight'])) {
        $elements[0]['_weight']['#access'] = FALSE;
      }

      // Remove table headers if they exist.
      if (isset($elements['#header'])) {
        unset($elements['#header']);
      }

      // Disable tabledrag.
      if (isset($elements['#tabledrag'])) {
        unset($elements['#tabledrag']);
      }
    }

    // Add pending invitations section OUTSIDE of field widget items.
    // This prevents it from being affected by mode change or field operation.
    if ($this->getSetting('show_pending_invites')) {
      $elements['pending_invites_section'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Pending Invitations'),
        '#weight' => 1000,
        '#attributes' => ['class' => ['pending-invitations-section']],
        'list' => $this->renderPendingInvites($items),
      ];
    }

    // Add current users section OUTSIDE of field widget items.
    $elements['current_users_section'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Current Users'),
      '#weight' => 999,
      '#attributes' => ['class' => ['current-users-section']],
      'list' => $this->renderExistingUsers($items),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    // If invitations are not enabled, use parent widget.
    if (!$this->getSetting('enable_invitations')) {
      return parent::formElement($items, $delta, $element, $form, $form_state);
    }

    // Attach library for widget styling and behavior.
    $element['#attached']['library'][] = 'user_reference_invite/widget';

    // Get parent autocomplete element.
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Add wrapper for our enhancements.
    // Use a consistent ID based on field name and delta for
    // AJAX to work properly.
    $field_name = $this->fieldDefinition->getName();
    $wrapper_id = Html::getId('user-invite-wrapper-' . $field_name . '-' . $delta);
    $element['#prefix'] = '<div id="' . $wrapper_id . '" class="user-invite-field-wrapper">';
    $element['#suffix'] = '</div>';

    // Add mode selector.
    // Check user input first (for AJAX), then form values, then default.
    $field_name = $this->fieldDefinition->getName();
    $user_input = $form_state->getUserInput();
    $default_mode = 'existing';

    if (isset($user_input[$field_name][$delta]['mode'])) {
      $default_mode = $user_input[$field_name][$delta]['mode'];
    }
    elseif ($form_state->getValue([$field_name, $delta, 'mode'])) {
      $default_mode = $form_state->getValue([$field_name, $delta, 'mode']);
    }

    // Only show mode selector on first delta (delta 0).
    // For invite mode, we'll handle multiple emails in one field.
    if ($delta === 0) {
      // Get selection modes setting.
      $selection_modes = $this->getSetting('selection_modes');

      // Build mode options based on configuration.
      $mode_options = [];
      if ($selection_modes === 'both') {
        $mode_options['existing'] = $this->t('Select existing user');
        $mode_options['invite'] = $this->t('Invite new users by email');
      }
      elseif ($selection_modes === 'invite_only') {
        $mode_options['invite'] = $this->t('Invite new users by email');
        // Force default mode to 'invite' if only invite is available.
        $default_mode = 'invite';
      }

      // Only show mode selector if there are multiple options.
      if (count($mode_options) > 1) {
        $element['mode'] = [
          '#type' => 'radios',
          '#title' => $this->t('Selection mode'),
          '#options' => $mode_options,
          '#default_value' => $default_mode,
          '#ajax' => [
            'callback' => [$this, 'updateFormElement'],
            'wrapper' => $wrapper_id,
            'effect' => 'fade',
          ],
          '#weight' => -100,
        ];
      }
      else {
        // Only one mode available, use it as hidden value.
        $element['mode'] = [
          '#type' => 'value',
          '#value' => $default_mode,
        ];
      }
    }
    else {
      // For delta > 0, hide mode selector and use the mode from delta 0.
      $element['mode'] = [
        '#type' => 'value',
        '#value' => $default_mode,
      ];
    }

    // Show different fields based on mode.
    if ($default_mode === 'invite') {
      // Hide the autocomplete field.
      $element['target_id']['#access'] = FALSE;

      // Only show email field on first delta (delta 0).
      if ($delta === 0) {
        // Add email field with textarea for multiple emails.
        $element['invite_email'] = [
          '#type' => 'textarea',
          '#title' => $this->t('Email addresses'),
          '#description' => $this->t('Enter one or more email addresses separated by commas. For example: user1@example.com, user2@example.com, user3@example.com'),
          '#required' => $element['#required'] ?? FALSE,
          '#weight' => -50,
          '#rows' => 3,
        ];

        // Add roles selection based on configuration.
        $role_config = $this->getEffectiveRoleConfig($items->getEntity());

        if ($role_config['show_role_assignment']) {
          // Show role checkboxes.
          $element['invite_roles'] = [
            '#type' => 'checkboxes',
            '#title' => $this->t('Assign roles'),
            '#description' => $this->t('Select roles to assign to all invited users.'),
            '#options' => $role_config['allowed_roles'],
            '#weight' => -40,
          ];
        }
        else {
          // Hide role field, store default roles.
          $element['invite_roles'] = [
            '#type' => 'value',
            '#value' => $role_config['default_roles'],
          ];
        }
      }
      else {
        // Hide additional deltas when in invite mode.
        $element['#access'] = FALSE;
      }
    }

    return $element;
  }

  /**
   * AJAX callback to update form element.
   */
  public function updateFormElement(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    // Get the field item wrapper (need to go up 2
    // levels: radio button -> mode fieldset -> field item).
    $parents = array_slice($triggering_element['#array_parents'], 0, -2);

    // Navigate to the element that needs to be returned.
    $element = $form;
    foreach ($parents as $parent) {
      $element = $element[$parent];
    }

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $entity = $form_state->getFormObject()->getEntity();

    // Get invitation manager service.
    $invite_manager = \Drupal::service('user_reference_invite.invite_manager');
    $messenger = \Drupal::messenger();

    // Clear tempstore to remove stale invitations and prevent duplicates.
    $tempstore = \Drupal::service('tempstore.private')->get('user_reference_invite');
    $tempstore->delete('pending_invites');

    foreach ($values as $delta => $value) {
      // Check if this is an invitation.
      if (isset($value['mode']) && $value['mode'] === 'invite' && !empty($value['invite_email'])) {
        try {
          // Get roles (only from delta 0 since that's where they're defined).
          $roles = [];
          if (isset($value['invite_roles'])) {
            $roles = array_filter($value['invite_roles']);
          }

          // Parse comma-separated email addresses.
          $email_string = trim($value['invite_email']);
          $emails = array_map('trim', explode(',', $email_string));

          // Filter out empty emails.
          $emails = array_filter($emails);

          // Track success/failure counts.
          $success_count = 0;
          $failed_emails = [];

          // Create invitation for each email.
          foreach ($emails as $email) {
            // Validate email format.
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
              $failed_emails[] = $email;
              $messenger->addError(
                    $this->t('Invalid email address: @email', ['@email' => $email])
                );
              continue;
            }

            try {
              // Check if entity is new (not yet saved).
              if ($entity->isNew()) {
                // Store invitation data in tempstore to process after save.
                $tempstore = \Drupal::service('tempstore.private')->get('user_reference_invite');
                $pending_invites = $tempstore->get('pending_invites') ?: [];
                $pending_invites[] = [
                  'email' => $email,
                  'entity_type' => $entity->getEntityTypeId(),
                  'field_name' => $this->fieldDefinition->getName(),
                  'roles' => array_values($roles),
                  'metadata' => ['delta' => $delta],
                ];
                $tempstore->set('pending_invites', $pending_invites);
                $success_count++;
              }
              else {
                // Entity already saved, create invitation immediately.
                // Check if invitation already exists for
                // this email/entity/field combination.
                $existing_invites = $invite_manager->findPendingInvitations(
                      $entity->getEntityTypeId(),
                      $entity->id(),
                      $this->fieldDefinition->getName(),
                      $email
                  );

                if (!empty($existing_invites)) {
                  // Invitation already exists, skip.
                  $messenger->addWarning(
                            $this->t(
                  'An invitation for @email already exists for this @entity_type.',
                  [
                    '@email' => $email,
                    '@entity_type' => $entity->getEntityType()->getLabel(),
                  ]
                            )
                        );
                  continue;
                }

                $metadata = [
                  'delta' => $delta,
                ];

                $invite = $invite_manager->createInvitation(
                      $email,
                      $entity->getEntityTypeId(),
                      $entity->id(),
                      $this->fieldDefinition->getName(),
                      array_values($roles),
                      $metadata
                  );

                // Send email.
                if ($invite_manager->sendInvitationEmail($invite)) {
                  $success_count++;
                }
                else {
                  $messenger->addWarning(
                        $this->t(
                            'Invitation created but email could not be sent to @email',
                            ['@email' => $email]
                        )
                    );
                }
              }
            }
            catch (\Exception $e) {
              $failed_emails[] = $email;
              $messenger->addError(
                    $this->t(
                        'Failed to create invitation for @email: @message',
                        [
                          '@email' => $email,
                          '@message' => $e->getMessage(),
                        ]
                    )
                );
            }
          }

          // Show summary message.
          if ($success_count > 0) {
            $messenger->addStatus(
                  $this->formatPlural(
                      $success_count,
                      'Invitation sent to 1 user.',
                      'Invitations sent to @count users.'
                  )
              );
          }

          // Clear invitation-specific fields but preserve existing target_id.
          // Allows existing references to remain while sending invitations.
          if (isset($values[$delta]['target_id']) && !empty($values[$delta]['target_id'])) {
            // Keep the existing user reference, just remove invitation fields.
            unset($values[$delta]['mode']);
            unset($values[$delta]['invite_email']);
            unset($values[$delta]['invite_roles']);
          }
          else {
            // No existing reference, remove the delta entirely.
            unset($values[$delta]);
          }
        }
        catch (\Exception $e) {
          $messenger->addError(
                $this->t(
                    'Failed to process invitations: @message',
                    ['@message' => $e->getMessage()]
                )
            );
          // On error, preserve target_id if it exists.
          if (isset($values[$delta]['target_id']) && !empty($values[$delta]['target_id'])) {
            unset($values[$delta]['mode']);
            unset($values[$delta]['invite_email']);
            unset($values[$delta]['invite_roles']);
          }
          else {
            unset($values[$delta]);
          }
        }
      }
    }

    // Ensure target_id is set for all values to prevent warnings from parent.
    // When in invite mode, the autocomplete field is hidden and target_id
    // doesn't exist in form values.
    foreach ($values as $delta => &$value) {
      if (!isset($value['target_id'])) {
        $value['target_id'] = NULL;
      }
    }

    return parent::massageFormValues($values, $form, $form_state);
  }

  /**
   * Render pending invitations for this field.
   */
  protected function renderPendingInvites(FieldItemListInterface $items) {
    $entity = $items->getEntity();

    // Can't show invites for new entities.
    if ($entity->isNew()) {
      return [
        '#markup' => '<p>' . $this->t('Save the entity first to send invitations.') . '</p>',
      ];
    }

    // Load pending invitations.
    $storage = \Drupal::entityTypeManager()->getStorage('user_invite');
    $ids = $storage->getQuery()
      ->condition('entity_type', $entity->getEntityTypeId())
      ->condition('entity_id', $entity->id())
      ->condition('field_name', $this->fieldDefinition->getName())
      ->condition('status', 'pending')
      ->accessCheck(FALSE)
      ->execute();

    if (empty($ids)) {
      return [
        '#markup' => '<p><em>' . $this->t('No pending invitations.') . '</em></p>',
      ];
    }

    $invites = $storage->loadMultiple($ids);

    // Get configured columns.
    $config = \Drupal::config('user_reference_invite.settings');
    $enabled_columns = $config->get('pending_invitations_table_columns');

    // Default columns if not configured.
    if (empty($enabled_columns)) {
      $enabled_columns = ['email', 'sent', 'expires'];
    }

    // Build table header and rows based on enabled columns.
    $available_columns = $this->getAvailablePendingInvitationsColumns();
    $header = [];
    foreach ($enabled_columns as $column_key) {
      if (isset($available_columns[$column_key])) {
        $header[] = $available_columns[$column_key];
      }
    }

    $rows = [];
    $date_formatter = \Drupal::service('date.formatter');
    foreach ($invites as $invite) {
      $row = [];
      foreach ($enabled_columns as $column_key) {
        $row[] = $this->getPendingInviteColumnValue($invite, $column_key, $date_formatter);
      }
      $rows[] = $row;
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No pending invitations.'),
    ];
  }

  /**
   * Render existing referenced users.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $items
   *   The field items.
   *
   * @return array
   *   Render array for existing users.
   */
  protected function renderExistingUsers(FieldItemListInterface $items) {
    $rows = [];

    // Get configured columns.
    $config = \Drupal::config('user_reference_invite.settings');
    $enabled_columns = $config->get('current_users_table_columns');

    // Default columns if not configured.
    if (empty($enabled_columns)) {
      $enabled_columns = ['name', 'email', 'roles'];
    }

    // Build table header based on enabled columns.
    $available_columns = $this->getAvailableCurrentUsersColumns();
    $header = [];
    foreach ($enabled_columns as $column_key) {
      if (isset($available_columns[$column_key])) {
        $header[] = $available_columns[$column_key];
      }
    }

    // Get all referenced user IDs from the field.
    $date_formatter = \Drupal::service('date.formatter');
    foreach ($items as $item) {
      if (!empty($item->target_id)) {
        $user = \Drupal::entityTypeManager()->getStorage('user')->load($item->target_id);
        if ($user) {
          $row = [];
          foreach ($enabled_columns as $column_key) {
            $row[] = $this->getCurrentUserColumnValue($user, $column_key, $date_formatter);
          }
          $rows[] = $row;
        }
      }
    }

    if (empty($rows)) {
      return [
        '#markup' => '<p><em>' . $this->t('No users currently referenced.') . '</em></p>',
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
    ];
  }

  /**
   * Get effective role configuration for current entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface|null $entity
   *   The entity being edited.
   *
   * @return array
   *   Array with keys:
   *   - show_role_assignment: Whether to show role checkboxes
   *   - default_roles: Array of default role IDs when not showing checkboxes
   *   - allowed_roles: Array of allowed roles (role_id => label)
   */
  protected function getEffectiveRoleConfig($entity = NULL) {
    $module_settings = \Drupal::config('user_reference_invite.settings');

    // Get global settings.
    $show_assignment = $module_settings->get('allow_role_assignment_on_content');
    $global_default_roles = $module_settings->get('default_roles') ?? [];
    $global_allowed_roles = $module_settings->get('allowed_global_roles') ?? [];

    // Widget-level roles override global roles.
    // Check widget settings first.
    $widget_roles = array_filter($this->getSetting('allowed_roles'));
    if (!empty($widget_roles)) {
      // Widget has specific roles configured - use these (override global).
      // Use widget roles for both allowed and default roles.
      $allowed_role_ids = array_keys($widget_roles);
      $default_role_ids = array_keys($widget_roles);
    }
    else {
      // Widget has no roles configured - fall back to global.
      $allowed_role_ids = $global_allowed_roles;
      $default_role_ids = $global_default_roles;
    }

    return [
      'show_role_assignment' => $show_assignment,
      'default_roles' => $default_role_ids,
      'allowed_roles' => $this->loadRoleOptions($allowed_role_ids),
    ];
  }

  /**
   * Load role options by role IDs.
   *
   * @param array $role_ids
   *   Array of role IDs.
   *
   * @return array
   *   Array of role_id => label.
   */
  protected function loadRoleOptions(array $role_ids) {
    if (empty($role_ids)) {
      // If no specific roles configured, load all except anonymous.
      $role_storage = \Drupal::entityTypeManager()->getStorage('user_role');
      $roles = [];
      foreach ($role_storage->loadMultiple() as $role_id => $role) {
        if ($role_id !== 'anonymous') {
          $roles[$role_id] = $role->label();
        }
      }
      return $roles;
    }

    // Load specific roles.
    $role_storage = \Drupal::entityTypeManager()->getStorage('user_role');
    $role_entities = $role_storage->loadMultiple($role_ids);
    $roles = [];
    foreach ($role_entities as $role_id => $role) {
      $roles[$role_id] = $role->label();
    }
    return $roles;
  }

  /**
   * Get available columns for the current users table.
   *
   * @return array
   *   Array of column machine names and labels.
   */
  protected function getAvailableCurrentUsersColumns() {
    return [
      'name' => $this->t('Name'),
      'email' => $this->t('Email'),
      'roles' => $this->t('Roles'),
      'status' => $this->t('Status'),
      'last_access' => $this->t('Last Access'),
    ];
  }

  /**
   * Get available columns for the pending invitations table.
   *
   * @return array
   *   Array of column machine names and labels.
   */
  protected function getAvailablePendingInvitationsColumns() {
    return [
      'email' => $this->t('Email'),
      'sent' => $this->t('Sent'),
      'expires' => $this->t('Expires'),
      'invited_by' => $this->t('Invited By'),
      'roles' => $this->t('Roles'),
    ];
  }

  /**
   * Get column value for a current user.
   *
   * @param \Drupal\user\UserInterface $user
   *   The user entity.
   * @param string $column_key
   *   The column key.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   *
   * @return string
   *   The column value.
   */
  protected function getCurrentUserColumnValue($user, $column_key, $date_formatter) {
    switch ($column_key) {
      case 'name':
        return $user->getDisplayName();

      case 'email':
        return $user->getEmail();

      case 'roles':
        $roles = array_filter(
              $user->getRoles(), function ($role) {
                  return $role !== 'authenticated';
              }
          );
        $role_labels = array_map(
              function ($role) {
                  $role_entity = \Drupal::entityTypeManager()
                    ->getStorage('user_role')
                    ->load($role);
                  return $role_entity ? $role_entity->label() : $role;
              }, $roles
          );
        return implode(', ', $role_labels);

      case 'status':
        return $user->isActive() ? $this->t('Active') : $this->t('Blocked');

      case 'last_access':
        $last_access = $user->getLastAccessedTime();
        return $last_access ? $date_formatter->format($last_access, 'short') : $this->t('Never');

      default:
        return '';
    }
  }

  /**
   * Get column value for a pending invitation.
   *
   * @param \Drupal\user_reference_invite\Entity\UserInviteInterface $invite
   *   The invitation entity.
   * @param string $column_key
   *   The column key.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   *
   * @return string
   *   The column value.
   */
  protected function getPendingInviteColumnValue($invite, $column_key, $date_formatter) {
    switch ($column_key) {
      case 'email':
        return $invite->getEmail();

      case 'sent':
        return $date_formatter->format($invite->getCreatedTime(), 'short');

      case 'expires':
        return $date_formatter->format($invite->getExpiresTime(), 'short');

      case 'invited_by':
        $inviter = \Drupal::entityTypeManager()
          ->getStorage('user')
          ->load($invite->getInvitedBy());
        return $inviter ? $inviter->getDisplayName() : $this->t('Unknown');

      case 'roles':
        $roles = $invite->get('roles')->value;
        if ($roles) {
          $role_array = json_decode($roles, TRUE);
          return $role_array ? implode(', ', $role_array) : '-';
        }
        return '-';

      default:
        return '';
    }
  }

}
