<?php

namespace Drupal\user_reference_invite\Service;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Database\Connection;

/**
 * Service for generating and validating invitation tokens.
 */
class TokenService implements TokenServiceInterface {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a TokenService object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public function generateToken(string $email, array $context = []): string {
    // Generate cryptographically secure random token.
    $token = Crypt::randomBytesBase64(32);

    // Ensure uniqueness.
    $attempts = 0;
    while ($this->tokenExists($this->hashToken($token)) && $attempts < 10) {
      $token = Crypt::randomBytesBase64(32);
      $attempts++;
    }

    return $token;
  }

  /**
   * {@inheritdoc}
   */
  public function hashToken(string $token): string {
    return hash('sha256', $token);
  }

  /**
   * {@inheritdoc}
   */
  public function verifyToken(string $token, string $hash): bool {
    return hash_equals($hash, $this->hashToken($token));
  }

  /**
   * Check if a token hash already exists in the database.
   *
   * @param string $hash
   *   The token hash.
   *
   * @return bool
   *   TRUE if exists, FALSE otherwise.
   */
  protected function tokenExists(string $hash): bool {
    $exists = $this->database->select('user_invite', 'ui')
      ->fields('ui', ['id'])
      ->condition('token', $hash)
      ->range(0, 1)
      ->execute()
      ->fetchField();

    return (bool) $exists;
  }

}
