<?php

namespace Drupal\user_reference_invite\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use Drupal\user_reference_invite\Entity\UserInviteInterface;
use Egulias\EmailValidator\EmailValidator;

/**
 * Service for managing user invitations.
 */
class UserInviteManager implements UserInviteManagerInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The token service.
   *
   * @var \Drupal\user_reference_invite\Service\TokenServiceInterface
   */
  protected $tokenService;

  /**
   * The email validator.
   *
   * @var \Egulias\EmailValidator\EmailValidator
   */
  protected $emailValidator;

  /**
   * The mail manager.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructs a UserInviteManager object.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    TokenServiceInterface $token_service,
    EmailValidator $email_validator,
    MailManagerInterface $mail_manager,
    LoggerChannelFactoryInterface $logger_factory,
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    TimeInterface $time,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->tokenService = $token_service;
    $this->emailValidator = $email_validator;
    $this->mailManager = $mail_manager;
    $this->logger = $logger_factory->get('user_reference_invite');
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public function createInvitation(
    string $email,
    string $entity_type,
    int $entity_id,
    string $field_name,
    array $roles = [],
    array $metadata = [],
  ): UserInviteInterface {
    // Validate email.
    if (!$this->emailValidator->isValid($email)) {
      throw new \InvalidArgumentException('Invalid email address');
    }

    // Generate token.
    $plain_token = $this->tokenService->generateToken($email, [
      'entity_type' => $entity_type,
      'entity_id' => $entity_id,
      'field_name' => $field_name,
    ]);
    $hashed_token = $this->tokenService->hashToken($plain_token);

    // Get expiry time.
    $config = $this->configFactory->get('user_reference_invite.settings');
    $expiry_days = $config->get('default_expiry_days') ?? 30;
    $expires = $this->time->getRequestTime() + ($expiry_days * 86400);

    // Create invitation entity.
    $invite = $this->entityTypeManager->getStorage('user_invite')->create([
      'email' => $email,
      'token' => $hashed_token,
      'status' => 'pending',
      'entity_type' => $entity_type,
      'entity_id' => $entity_id,
      'field_name' => $field_name,
      'roles' => json_encode($roles),
      'invited_by' => $this->currentUser->id(),
      'created' => $this->time->getRequestTime(),
      'expires' => $expires,
      'metadata' => json_encode($metadata),
    ]);

    $invite->save();

    // Store plain token temporarily for email (not saved to DB).
    $invite->plain_token = $plain_token;

    $this->logger->info('Invitation created for @email by user @uid', [
      '@email' => $email,
      '@uid' => $this->currentUser->id(),
    ]);

    return $invite;
  }

  /**
   * {@inheritdoc}
   */
  public function sendInvitationEmail(UserInviteInterface $invite): bool {
    $config = $this->configFactory->get('user_reference_invite.settings');

    if (!$config->get('send_invitation_email')) {
      return FALSE;
    }

    // Get plain token (must be set immediately after creation).
    $plain_token = $invite->plain_token ?? NULL;
    if (!$plain_token) {
      $this->logger->error('Cannot send invitation email: plain token not available');
      return FALSE;
    }

    // Load inviter.
    $inviter = $this->entityTypeManager->getStorage('user')->load($invite->getInvitedBy());

    // Load target entity for context.
    $target_entity = $this->entityTypeManager
      ->getStorage($invite->getTargetEntityType())
      ->load($invite->getTargetEntityId());

    // Check if user already exists.
    $existing_users = $this->entityTypeManager->getStorage('user')
      ->loadByProperties(['mail' => $invite->getEmail()]);
    $user_exists = !empty($existing_users);

    // Build URLs.
    $accept_url = Url::fromRoute('user_reference_invite.accept', [
      'token' => $plain_token,
    ], ['absolute' => TRUE])->toString();

    $register_url = Url::fromRoute('user.register', [], [
      'absolute' => TRUE,
      'query' => ['invite_token' => $plain_token],
    ])->toString();

    $login_url = Url::fromRoute('user.login', [], [
      'absolute' => TRUE,
      'query' => ['invite_token' => $plain_token],
    ])->toString();

    // Prepare email parameters.
    $params = [
      'invite' => $invite,
      'invited_by' => $inviter ? $inviter->getDisplayName() : 'Site Administrator',
      'entity_label' => $target_entity ? $target_entity->label() : 'the system',
      'register_url' => $register_url,
      'login_url' => $login_url,
      'accept_url' => $accept_url,
      'expiry_date' => \Drupal::service('date.formatter')->format($invite->getExpiresTime(), 'medium'),
      'roles' => json_decode($invite->get('roles')->value, TRUE) ?? [],
      'user_exists' => $user_exists,
    ];

    // Send email.
    $result = $this->mailManager->mail(
      'user_reference_invite',
      'user_invitation',
      $invite->getEmail(),
      \Drupal::languageManager()->getDefaultLanguage()->getId(),
      $params
    );

    if ($result['result']) {
      $this->logger->info('Invitation email sent to @email', [
        '@email' => $invite->getEmail(),
      ]);
      return TRUE;
    }

    $this->logger->error('Failed to send invitation email to @email', [
      '@email' => $invite->getEmail(),
    ]);
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function acceptInvitation(string $token, UserInterface $user): bool {
    // Find invitation by token.
    $hashed_token = $this->tokenService->hashToken($token);
    $invites = $this->entityTypeManager->getStorage('user_invite')
      ->loadByProperties(['token' => $hashed_token]);

    if (empty($invites)) {
      $this->logger->warning('Invalid invitation token');
      return FALSE;
    }

    $invite = reset($invites);

    // Check if already accepted.
    if ($invite->getStatus() === 'accepted') {
      return TRUE;
    }

    // Check if expired.
    if ($invite->isExpired()) {
      $invite->setStatus('expired');
      $invite->save();
      $this->logger->warning('Invitation expired for @email', [
        '@email' => $invite->getEmail(),
      ]);
      return FALSE;
    }

    // Mark as accepted.
    $invite->setStatus('accepted');
    $invite->set('accepted', $this->time->getRequestTime());
    $invite->set('accepted_by', $user->id());
    $invite->save();

    $this->logger->info('Invitation accepted by @email (uid: @uid)', [
      '@email' => $invite->getEmail(),
      '@uid' => $user->id(),
    ]);

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function attachUserToField(UserInviteInterface $invite, int $user_id): bool {
    try {
      // Load target entity.
      $entity = $this->entityTypeManager
        ->getStorage($invite->getTargetEntityType())
        ->load($invite->getTargetEntityId());

      if (!$entity) {
        $this->logger->error('Target entity not found for invitation @id', [
          '@id' => $invite->id(),
        ]);
        return FALSE;
      }

      $field_name = $invite->getFieldName();
      $field_safe = is_scalar($field_name) ? (string) $field_name : 'unknown';

      // Check if field exists.
      if (!$entity->hasField($field_name)) {
        $this->logger->error('Field @field not found on entity', [
          '@field' => $field_name,
        ]);
        return FALSE;
      }

      // Get current field values.
      $field_values = $entity->get($field_name)->getValue();

      // Check if user already in field.
      foreach ($field_values as $value) {
        if (isset($value['target_id']) && $value['target_id'] == $user_id) {
          // Already attached.
          return TRUE;
        }
      }

      // Add user to field.
      $field_values[] = ['target_id' => $user_id];
      $entity->set($field_name, $field_values);
      $entity->save();

      $this->logger->info('User @uid attached to field @field on @entity_type:@entity_id', [
        '@uid' => $user_id,
        '@field' => $field_name,
        '@entity_type' => $invite->getTargetEntityType(),
        '@entity_id' => $invite->getTargetEntityId(),
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to attach user to field: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function findPendingInvitationsByEmail(string $email): array {
    $storage = $this->entityTypeManager->getStorage('user_invite');
    $ids = $storage->getQuery()
      ->condition('email', $email)
      ->condition('status', 'pending')
      ->accessCheck(FALSE)
      ->execute();

    if (empty($ids)) {
      return [];
    }

    $invites = $storage->loadMultiple($ids);

    // Filter out expired.
    $pending = [];
    foreach ($invites as $invite) {
      if (!$invite->isExpired()) {
        $pending[] = $invite;
      }
      else {
        // Mark as expired.
        $invite->setStatus('expired');
        $invite->save();
      }
    }

    return $pending;
  }

  /**
   * {@inheritdoc}
   */
  public function findPendingInvitations(
    string $entity_type,
    int $entity_id,
    string $field_name,
    string $email,
  ): array {
    $storage = $this->entityTypeManager->getStorage('user_invite');
    $ids = $storage->getQuery()
      ->condition('email', $email)
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->condition('field_name', $field_name)
      ->condition('status', 'pending')
      ->accessCheck(FALSE)
      ->execute();

    if (empty($ids)) {
      return [];
    }

    $invites = $storage->loadMultiple($ids);

    // Filter out expired.
    $pending = [];
    foreach ($invites as $invite) {
      if (!$invite->isExpired()) {
        $pending[] = $invite;
      }
      else {
        // Mark as expired.
        $invite->setStatus('expired');
        $invite->save();
      }
    }

    return $pending;
  }

  /**
   * {@inheritdoc}
   */
  public function cleanupExpiredInvitations(): int {
    $config = $this->configFactory->get('user_reference_invite.settings');
    $batch_size = $config->get('cleanup_batch_size') ?? 100;

    $storage = $this->entityTypeManager->getStorage('user_invite');

    // Find expired pending invitations.
    $ids = $storage->getQuery()
      ->condition('status', 'pending')
      ->condition('expires', $this->time->getRequestTime(), '<')
      ->range(0, $batch_size)
      ->accessCheck(FALSE)
      ->execute();

    $count = 0;

    if ($ids) {
      $invites = $storage->loadMultiple($ids);
      foreach ($invites as $invite) {
        $invite->setStatus('expired');
        $invite->save();
        $count++;
      }
    }

    // Delete old accepted invitations if configured.
    $cleanup_days = $config->get('cleanup_accepted_after_days');
    if ($cleanup_days) {
      $cutoff = $this->time->getRequestTime() - ($cleanup_days * 86400);
      $old_ids = $storage->getQuery()
        ->condition('status', 'accepted')
        ->condition('accepted', $cutoff, '<')
        ->range(0, $batch_size)
        ->accessCheck(FALSE)
        ->execute();

      if ($old_ids) {
        $old_invites = $storage->loadMultiple($old_ids);
        $storage->delete($old_invites);
        $count += count($old_invites);
      }
    }

    return $count;
  }

  /**
   * {@inheritdoc}
   */
  public function validateToken(string $token): ?UserInviteInterface {
    // Hash the token to match stored format.
    $hashed_token = $this->tokenService->hashToken($token);

    // Find invitation by token.
    $invites = $this->entityTypeManager->getStorage('user_invite')
      ->loadByProperties(['token' => $hashed_token]);

    if (empty($invites)) {
      return NULL;
    }

    $invite = reset($invites);

    // Check if already accepted.
    if ($invite->getStatus() === 'accepted') {
      return NULL;
    }

    // Check if expired.
    if ($invite->isExpired()) {
      $invite->setStatus('expired');
      $invite->save();
      return NULL;
    }

    // Return valid pending invitation.
    return $invite;
  }

}
