<?php

namespace Drupal\user_reference_invite\Service;

use Drupal\user\UserInterface;
use Drupal\user_reference_invite\Entity\UserInviteInterface;

/**
 * Interface for the User Invite Manager Service.
 */
interface UserInviteManagerInterface {

  /**
   * Create a new user invitation.
   *
   * @param string $email
   *   The email address to invite.
   * @param string $entity_type
   *   The entity type containing the field.
   * @param int $entity_id
   *   The entity ID.
   * @param string $field_name
   *   The field name.
   * @param array $roles
   *   Context-specific roles to assign.
   * @param array $metadata
   *   Additional context data.
   *
   * @return \Drupal\user_reference_invite\Entity\UserInviteInterface
   *   The created invitation entity.
   *
   * @throws \Exception
   */
  public function createInvitation(
    string $email,
    string $entity_type,
    int $entity_id,
    string $field_name,
    array $roles = [],
    array $metadata = [],
  ): UserInviteInterface;

  /**
   * Send invitation email.
   *
   * @param \Drupal\user_reference_invite\Entity\UserInviteInterface $invite
   *   The invitation entity.
   *
   * @return bool
   *   TRUE if sent successfully.
   */
  public function sendInvitationEmail(UserInviteInterface $invite): bool;

  /**
   * Accept an invitation.
   *
   * @param string $token
   *   The invitation token.
   * @param \Drupal\user\UserInterface $user
   *   The user accepting the invitation.
   *
   * @return bool
   *   TRUE if accepted successfully.
   */
  public function acceptInvitation(string $token, UserInterface $user): bool;

  /**
   * Attach user to the original reference field.
   *
   * @param \Drupal\user_reference_invite\Entity\UserInviteInterface $invite
   *   The invitation entity.
   * @param int $user_id
   *   The user ID to attach.
   *
   * @return bool
   *   TRUE if attached successfully.
   */
  public function attachUserToField(UserInviteInterface $invite, int $user_id): bool;

  /**
   * Find pending invitations by email.
   *
   * @param string $email
   *   The email address.
   *
   * @return \Drupal\user_reference_invite\Entity\UserInviteInterface[]
   *   Array of pending invitations.
   */
  public function findPendingInvitationsByEmail(string $email): array;

  /**
   * Find pending invitations for a specific entity, field, and email.
   *
   * @param string $entity_type
   *   The entity type.
   * @param int $entity_id
   *   The entity ID.
   * @param string $field_name
   *   The field name.
   * @param string $email
   *   The email address.
   *
   * @return \Drupal\user_reference_invite\Entity\UserInviteInterface[]
   *   Array of pending invitations matching the criteria.
   */
  public function findPendingInvitations(
    string $entity_type,
    int $entity_id,
    string $field_name,
    string $email,
  ): array;

  /**
   * Clean up expired invitations.
   *
   * @return int
   *   Number of invitations cleaned up.
   */
  public function cleanupExpiredInvitations(): int;

  /**
   * Validate an invitation token.
   *
   * @param string $token
   *   The invitation token to validate.
   *
   * @return \Drupal\user_reference_invite\Entity\UserInviteInterface|null
   *   The invitation entity if valid and not expired, NULL otherwise.
   */
  public function validateToken(string $token): ?UserInviteInterface;

}
