<?php

namespace Drupal\user_reference_invite\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for admin invitation management.
 */
class InviteAdminController extends ControllerBase {

  /**
   * Lists all invitations.
   */
  public function listInvitations(Request $request) {
    $storage = $this->entityTypeManager()->getStorage('user_invite');

    // Build filter form.
    $current_status = $request->query->get('status', '');
    $current_entity_id = $request->query->get('entity_id', '');

    $filter_form = [
      '#type' => 'container',
      '#attributes' => ['class' => ['invitation-filters']],
    ];

    $filter_form['filters'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form--inline', 'clearfix']],
    ];

    $filter_form['filters']['status'] = [
      '#type' => 'select',
      '#title' => $this->t('Status'),
      '#options' => [
        '' => $this->t('- All -'),
        'pending' => $this->t('Pending'),
        'accepted' => $this->t('Accepted'),
        'expired' => $this->t('Expired'),
        'canceled' => $this->t('Canceled'),
      ],
      '#default_value' => $current_status,
      '#attributes' => ['class' => ['form-item']],
    ];

    $filter_form['filters']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['form-actions']],
    ];

    $filter_form['filters']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Filter'),
      '#button_type' => 'primary',
      '#submit' => ['::filterSubmit'],
    ];

    if ($current_status) {
      $filter_form['filters']['actions']['reset'] = [
        '#type' => 'link',
        '#title' => $this->t('Reset'),
        '#url' => Url::fromRoute('user_reference_invite.admin.list'),
        '#attributes' => ['class' => ['button']],
      ];
    }

    // Wrap filter in a form.
    $filter_form['#method'] = 'get';
    $filter_form['#action'] = Url::fromRoute('user_reference_invite.admin.list')->toString();

    // Build query.
    $query = $storage->getQuery()
      ->accessCheck(TRUE)
      ->sort('created', 'DESC')
      ->pager(50);

    // Apply filters.
    if ($current_status) {
      $query->condition('status', $current_status);
    }
    if ($current_entity_id) {
      $query->condition('entity_id', $current_entity_id);
    }

    $ids = $query->execute();
    $invites = $storage->loadMultiple($ids);

    // Get configured columns.
    $config = $this->config('user_reference_invite.settings');
    $enabled_columns = $config->get('admin_table_columns');

    // Default columns if not configured.
    if (empty($enabled_columns)) {
      $enabled_columns = [
        'email',
        'status',
        'entity',
        'field',
        'invited_by',
        'created',
        'expires',
      ];
    }

    // Build table header based on enabled columns.
    $available_columns = $this->getAvailableColumns();
    $header = [];
    foreach ($enabled_columns as $column_key) {
      if (isset($available_columns[$column_key])) {
        $header[] = $available_columns[$column_key];
      }
    }

    $rows = [];
    foreach ($invites as $invite) {
      $row = [];

      foreach ($enabled_columns as $column_key) {
        $row[] = $this->getColumnValue($invite, $column_key);
      }

      $rows[] = $row;
    }

    $build = [];

    // Add filter form using proper markup with inline styling.
    $filter_markup = '<div class="invitation-filters-wrapper" style="background: #f5f5f5; padding: 15px; margin-bottom: 20px; border: 1px solid #ddd; border-radius: 4px;">';
    $filter_markup .= '<form method="get" action="' . Url::fromRoute('user_reference_invite.admin.list')->toString() . '" class="invitation-filters-form" style="display: flex; gap: 10px; align-items: flex-end; flex-wrap: wrap;">';

    $filter_markup .= '<div class="form-item" style="margin: 0; min-width: 180px; flex: 1 1 auto; max-width: 250px;">';
    $filter_markup .= '<label for="edit-status" style="display: block; margin-bottom: 5px; font-weight: bold; font-size: 14px;">' . $this->t('Status') . '</label>';
    $filter_markup .= '<select name="status" id="edit-status" class="form-select" style="width: 100%; padding: 8px; border: 1px solid #ccc; border-radius: 4px; font-size: 14px; box-sizing: border-box;">';
    $filter_markup .= '<option value=""' . ($current_status === '' ? ' selected' : '') . '>' . $this->t('- All -') . '</option>';
    $filter_markup .= '<option value="pending"' . ($current_status === 'pending' ? ' selected' : '') . '>' . $this->t('Pending') . '</option>';
    $filter_markup .= '<option value="accepted"' . ($current_status === 'accepted' ? ' selected' : '') . '>' . $this->t('Accepted') . '</option>';
    $filter_markup .= '<option value="expired"' . ($current_status === 'expired' ? ' selected' : '') . '>' . $this->t('Expired') . '</option>';
    $filter_markup .= '<option value="canceled"' . ($current_status === 'canceled' ? ' selected' : '') . '>' . $this->t('Canceled') . '</option>';
    $filter_markup .= '</select>';
    $filter_markup .= '</div>';

    $filter_markup .= '<div class="form-item" style="margin: 0; min-width: 180px; flex: 1 1 auto; max-width: 250px;">';
    $filter_markup .= '<label for="edit-entity-id" style="display: block; margin-bottom: 5px; font-weight: bold; font-size: 14px;">' . $this->t('Entity ID') . '</label>';
    $filter_markup .= $this->t(
      '<input type="text"
        name="entity_id"
        id="edit-entity-id"
        class="form-text"
        value=":value"
        placeholder=":placeholder"
        style="width: 100%; padding: 8px; border: 1px solid #ccc;
              border-radius: 4px; font-size: 14px; box-sizing: border-box;" />',
      [
        ':value' => $current_entity_id,
        ':placeholder' => $this->t('e.g., 123'),
      ]
    );
    $filter_markup .= '</div>';

    $filter_markup .= '<div class="form-actions" style="margin: 0; display: flex; gap: 8px; flex-shrink: 0; margin-top: 25px;">';
    $filter_markup .= '<button type="submit" class="button button--primary" style="padding: 8px 16px; background: #0073aa; color: white; border: none; border-radius: 4px; cursor: pointer; font-weight: bold; font-size: 14px; white-space: nowrap;">' . $this->t('Apply Filters') . '</button>';

    if ($current_status || $current_entity_id) {
      $reset_url = Url::fromRoute('user_reference_invite.admin.list')->toString();
      $filter_markup .= '<a href="' . $reset_url . '" class="button" style="padding: 8px 16px; background: #f0f0f0; color: #333; border: 1px solid #ccc; border-radius: 4px; text-decoration: none; display: inline-block; font-weight: bold; font-size: 14px; white-space: nowrap; line-height: 1.4;">' . $this->t('Clear Filters') . '</a>';
    }

    $filter_markup .= '</div>';
    $filter_markup .= '</form>';
    $filter_markup .= '</div>';

    $build['filters'] = [
      '#markup' => Markup::create($filter_markup),
    ];

    // Add pager at top.
    $build['pager_top'] = ['#type' => 'pager'];

    // Add table.
    $build['table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No invitations found.'),
    ];

    // Add pager at bottom.
    $build['pager_bottom'] = ['#type' => 'pager'];

    $build['#cache'] = [
      'tags' => ['user_invite_list'],
      'contexts' => ['url.query_args'],
    ];

    return $build;
  }

  /**
   * Get available columns for display.
   *
   * @return array
   *   Array of column machine names and labels.
   */
  protected function getAvailableColumns() {
    return [
      'email' => $this->t('Email'),
      'status' => $this->t('Status'),
      'entity' => $this->t('Entity'),
      'field' => $this->t('Field'),
      'invited_by' => $this->t('Invited By'),
      'created' => $this->t('Created'),
      'expires' => $this->t('Expires'),
      'accepted' => $this->t('Accepted Date'),
      'accepted_by' => $this->t('Accepted By'),
      'roles' => $this->t('Roles'),
    ];
  }

  /**
   * Get column value for an invite.
   *
   * @param \Drupal\user_reference_invite\Entity\UserInviteInterface $invite
   *   The invitation entity.
   * @param string $column_key
   *   The column key.
   *
   * @return string
   *   The column value.
   */
  protected function getColumnValue($invite, $column_key) {
    $date_formatter = \Drupal::service('date.formatter');

    switch ($column_key) {
      case 'email':
        return $invite->getEmail();

      case 'status':
        return ucfirst($invite->getStatus());

      case 'entity':
        return $invite->getTargetEntityType() . ':' . $invite->getTargetEntityId();

      case 'field':
        return $invite->getFieldName();

      case 'invited_by':
        $inviter = $this->entityTypeManager()
          ->getStorage('user')
          ->load($invite->getInvitedBy());
        return $inviter ? $inviter->getDisplayName() : 'Unknown';

      case 'created':
        return $date_formatter->format($invite->getCreatedTime(), 'short');

      case 'expires':
        return $date_formatter->format($invite->getExpiresTime(), 'short');

      case 'accepted':
        $accepted_time = $invite->get('accepted')->value;
        return $accepted_time ? $date_formatter->format($accepted_time, 'short') : '-';

      case 'accepted_by':
        $accepted_by_id = $invite->get('accepted_by')->target_id;
        if ($accepted_by_id) {
          $accepted_user = $this->entityTypeManager()
            ->getStorage('user')
            ->load($accepted_by_id);
          return $accepted_user ? $accepted_user->getDisplayName() : 'Unknown';
        }
        return '-';

      case 'roles':
        $roles = $invite->get('roles')->value;
        if ($roles) {
          $role_array = json_decode($roles, TRUE);
          return $role_array ? implode(', ', $role_array) : '-';
        }
        return '-';

      default:
        return '';
    }
  }

}
