<?php

namespace Drupal\user_register_notify\Form;

use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\token\TreeBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The module's settings form.
 */
class UserRegisterNotifyAdminSettingsForm extends ConfigFormBase {

  /**
   * The mail validator instance.
   *
   * @var \Drupal\Component\Utility\EmailValidatorInterface
   */
  protected $mailValidator;

  /**
   * The token tree builder instance.
   *
   * @var \Drupal\token\TreeBuilder
   */
  protected $tokenTreeBuilder;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Component\Utility\EmailValidatorInterface $mail_validator
   *   The mail validator instance.
   * @param \Drupal\token\TreeBuilder $token_tree_builder
   *   The token tree builder instance.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EmailValidatorInterface $mail_validator, TreeBuilder $token_tree_builder) {
    parent::__construct($config_factory);
    $this->mailValidator = $mail_validator;
    $this->tokenTreeBuilder = $token_tree_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('email.validator'),
      $container->get('token.tree_builder'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'user_register_notify_admin_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['user_register_notify.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $site_config = $this->config('system.site');
    $register_config = $this->config('user_register_notify.settings');
    $form['type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Registration notification type'),
      '#description' => $this->t('Where to send the registration notification to.'),
      '#default_value' => $register_config->get('type'),
      '#options' => [
        'custom' => $this->t('Send e-mail to custom e-mail address(es)'),
        'role' => $this->t('Send e-mail to specific role(s)'),
        'both' => $this->t('Send e-mail to both custom e-mail address(es) and specific role(s)'),
      ],
    ];
    $form['mail_to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom e-mail address(es)'),
      '#default_value' => $register_config->get('mail_to'),
      '#description' => $this->t('Comma separated list of e-mail addresses to be used as the "to" address for all user notifications listed below. Leave empty to use the default system e-mail address <em>(%site-email).</em>', [
        '%site-email' => $site_config->get('mail') ?? 'UNSET',
      ]),
      '#states' => [
        'visible' => [
          [
            ':input[name="type"]' => [
              'value' => 'custom',
            ],
          ],
          'or',
          [
            ':input[name="type"]' => [
              'value' => 'both',
            ],
          ],
        ],
      ],
    ];
    $form['roles'] = [
      '#default_value' => $register_config->get('roles') ?? [],
      '#description' => $this->t('All users with these checked roles will receive a notification e-mail when selected.'),
      '#options' => user_role_names(TRUE) ?? [],
      '#title' => $this->t('Send e-mail to specific role(s)'),
      '#type' => 'checkboxes',
      '#states' => [
        'visible' => [
          [
            ':input[name="type"]' => [
              'value' => 'role',
            ],
          ],
          'or',
          [
            ':input[name="type"]' => [
              'value' => 'both',
            ],
          ],
        ],
      ],
    ];
    $form['events'] = [
      '#title' => $this->t('Events'),
      '#type' => 'checkboxes',
      '#default_value' => $register_config->get('events') ?? [],
      '#description' => $this->t('When to send an e-mail to the specified roles / custom mails.'),
      '#options' => [
        'create' => $this->t('On user creation'),
        'update' => $this->t('On user update'),
        'delete' => $this->t('On user deletion'),
      ],
      '#required' => FALSE,
    ];
    $form['user_register_notify_created'] = [
      '#type' => 'details',
      '#title' => $this->t('Created mail'),
      '#open' => FALSE,
      '#description' => $this->t('Edit the notification e-mail message sent, if a new user user was created (Based on the selected roles and / or custom email adresses provided above).'),
      '#states' => [
        'visible' => [
          [
            ':input[name="events[create]"]' => ['checked' => TRUE],
          ],
        ],
      ],
    ];
    $form['user_register_notify_created']['created_subject'] = [
      '#default_value' => $register_config->get('created_subject'),
      '#description' => $this->t('Subject of the user created notification e-mail.'),
      '#required' => TRUE,
      '#title' => $this->t('Subject'),
      '#type' => 'textfield',
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['user'],
    ];
    $form['user_register_notify_created']['created_body'] = [
      '#default_value' => $register_config->get('created_body'),
      '#description' => $this->t('Customize the body of the user created notification e-mail.'),
      '#required' => TRUE,
      '#rows' => 10,
      '#title' => $this->t('Body'),
      '#type' => 'textarea',
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['user'],
    ];
    $form['user_register_notify_created']['created_token_tree'] = [
      '#theme' => 'token_tree_link',
      '#global_types' => TRUE,
      '#token_types' => ['user'],
      '#dialog' => TRUE,
      '#click_insert' => TRUE,
      '#show_restricted' => FALSE,
    ];
    $form['user_register_notify_created']['created_roles_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Send mails about the creation of users...'),
      '#options' => [
        'exclude' => $this->t('with any roles, except the selected'),
        'include' => $this->t('with the selected roles only'),
      ],
      '#default_value' => $register_config->get('created_roles_mode'),
    ];
    $form['user_register_notify_created']['created_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles'),
      '#options' => user_role_names(TRUE),
      '#default_value' => $register_config->get('created_roles'),
      '#description' => $this->t('If no roles are selected, mails about the creation of users are always sent.'),
    ];

    $form['user_register_notify_updated'] = [
      '#type' => 'details',
      '#title' => $this->t('Updated mail'),
      '#open' => FALSE,
      '#description' => $this->t('Edit the notification e-mail message sent, if a user user was updated (Based on the selected roles and / or custom email adresses provided above).'),
      '#states' => [
        'visible' => [
          [
            ':input[name="events[update]"]' => ['checked' => TRUE],
          ],
        ],
      ],
    ];
    $form['user_register_notify_updated']['updated_subject'] = [
      '#default_value' => $register_config->get('updated_subject'),
      '#description' => $this->t('Subject of user updated messages.'),
      '#required' => TRUE,
      '#title' => $this->t('Subject'),
      '#type' => 'textfield',
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['user'],
    ];
    $form['user_register_notify_updated']['updated_body'] = [
      '#default_value' => $register_config->get('updated_body'),
      '#description' => $this->t('Customize the body of the user updated notification e-mail.'),
      '#required' => TRUE,
      '#rows' => 10,
      '#title' => $this->t('Body'),
      '#type' => 'textarea',
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['user'],
    ];
    $form['user_register_notify_updated']['updated_token_tree'] = [
      '#theme' => 'token_tree_link',
      '#global_types' => TRUE,
      '#token_types' => ['user'],
      '#dialog' => TRUE,
      '#click_insert' => TRUE,
      '#show_restricted' => FALSE,
    ];
    $form['user_register_notify_updated']['updated_roles_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Send mails about the update of users...'),
      '#options' => [
        'exclude' => $this->t('with any roles, except the selected'),
        'include' => $this->t('with the selected roles only'),
      ],
      '#default_value' => $register_config->get('updated_roles_mode'),
    ];
    $form['user_register_notify_updated']['updated_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles'),
      '#options' => user_role_names(TRUE),
      '#default_value' => $register_config->get('updated_roles'),
      '#description' => $this->t('If no roles are selected, mails about the update of users are always sent.'),
    ];

    $form['user_register_notify_deleted'] = [
      '#type' => 'details',
      '#title' => $this->t('Deleted mail'),
      '#open' => FALSE,
      '#description' => $this->t('Edit the notification e-mail message sent, if a user  was deleted (Based on the selected roles and / or custom email adresses provided above).'),
      '#states' => [
        'visible' => [
          [
            ':input[name="events[delete]"]' => ['checked' => TRUE],
          ],
        ],
      ],
    ];
    $form['user_register_notify_deleted']['deleted_subject'] = [
      '#default_value' => $register_config->get('deleted_subject'),
      '#description' => $this->t('Subject of user deleted messages.'),
      '#required' => TRUE,
      '#title' => $this->t('Subject'),
      '#type' => 'textfield',
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['user'],
    ];
    $form['user_register_notify_deleted']['deleted_body'] = [
      '#default_value' => $register_config->get('deleted_body'),
      '#description' => $this->t('Customize the body of the user deleted notification e-mail.'),
      '#required' => TRUE,
      '#rows' => 10,
      '#title' => $this->t('Body'),
      '#type' => 'textarea',
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['user'],
    ];
    $form['user_register_notify_deleted']['deleted_token_tree'] = [
      '#theme' => 'token_tree_link',
      '#global_types' => TRUE,
      '#token_types' => ['user'],
      '#dialog' => TRUE,
      '#click_insert' => TRUE,
      '#show_restricted' => FALSE,
    ];
    $form['user_register_notify_deleted']['deleted_roles_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Send mails about the deletion of users...'),
      '#options' => [
        'exclude' => $this->t('with any roles, except the selected'),
        'include' => $this->t('with the selected roles only'),
      ],
      '#default_value' => $register_config->get('deleted_roles_mode'),
    ];
    $form['user_register_notify_deleted']['deleted_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles'),
      '#options' => user_role_names(TRUE),
      '#default_value' => $register_config->get('deleted_roles'),
      '#description' => $this->t('If no roles are selected, mails about the deletion of users are always sent.'),
    ];

    $form['risky_mail_header_modifications'] = [
      '#type' => 'details',
      '#title' => $this->t('Risky mail header modifications'),
      '#description' => $this->t('You should probably not set these unless you have specific requirements. In particular, modifying the "from" address could result in e-mail being blocked as spam, and the response could impact on your server.'),
    ];
    $form['risky_mail_header_modifications']['mail_from'] = [
      '#default_value' => $register_config->get('mail_from'),
      '#description' => $this->t('The e-mail address to be used as the "from" address for all user notifications listed below. Leave empty to use the default system e-mail address <em>(%site-email).</em>', [
        '%site-email' => $site_config->get('mail'),
      ]),
      '#title' => $this->t('"From" mail header'),
      '#type' => 'email',
    ];
    $form['risky_mail_header_modifications']['mail_reply_to'] = [
      '#default_value' => $register_config->get('mail_reply_to'),
      '#description' => $this->t('The e-mail address to be used as the "reply-to" address for all user notifications listed below. Leave empty for not using "reply-to" e-mail header. Configure this field only, if your "reply-to" address is not your "from" address.'),
      '#title' => $this->t('"Reply-to" mail header'),
      '#type' => 'email',
    ];
    $form['risky_mail_header_modifications']['mail_message_ids_header_overwrite_header'] = [
      '#title' => $this->t('System mail messages to overwrite'),
      '#description' => $this->t('The system mail messages to overwrite "From" and "Reply-to" on.'),
      '#default_value' => $register_config->get('mail_message_ids_header_overwrite_header') ?? [],
      '#options' => [
        // Our module mail ids:
        'user_register_notify_user_register_notify_create' => $this->t('Admin (user created)'),
        'user_register_notify_user_register_notify_update' => $this->t('Admin (user updated)'),
        'user_register_notify_user_register_notify_delete' => $this->t('Admin (user deleted)'),
        // Core - User module.
        'user_register_admin_created' => $this->t('Welcome (new user created by administrator)'),
        'user_register_no_approval_required' => $this->t('Welcome (no approval required)'),
        // If a user registered requiring admin approval, notify the admin, too.
        'user_register_pending_approval_admin' => $this->t('Admin (user awaiting approval)'),
        'user_register_pending_approval' => $this->t('Welcome (awaiting approval)'),
        'user_status_activated' => $this->t('User activation'),
        'user_status_blocked' => $this->t('User blocked'),
        'user_cancel_confirm' => $this->t('User cancellation confirmation'),
        'user_status_canceled' => $this->t('User canceled'),
        'user_password_reset' => $this->t('Password recovery'),
      ],
      '#type' => 'checkboxes',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Cleanup the e-mail string.
    $user_register_notify_mail_to = explode(',', trim($form_state->getValue('mail_to')));
    $form_state->setValue('mail_to', implode(',', $user_register_notify_mail_to));

    if (in_array($form_state->getValue('type'), ['custom', 'both'])) {
      foreach ($user_register_notify_mail_to as $email) {
        if (!$this->mailValidator->isValid($email)) {
          $form_state->setErrorByName('mail_to', $this->t('The e-mail address %mail is not valid.', [
            '%mail' => $email,
          ]));
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('user_register_notify.settings')
      ->set('type', $form_state->getValue('type'))
      ->set('mail_to', $form_state->getValue('mail_to'))
      ->set('roles', $form_state->getValue('roles'))
      ->set('created_subject', $form_state->getValue('created_subject'))
      ->set('created_body', $form_state->getValue('created_body'))
      ->set('updated_subject', $form_state->getValue('updated_subject'))
      ->set('updated_body', $form_state->getValue('updated_body'))
      ->set('deleted_subject', $form_state->getValue('deleted_subject'))
      ->set('deleted_body', $form_state->getValue('deleted_body'))
      ->set('mail_from', $form_state->getValue('mail_from'))
      ->set('mail_reply_to', $form_state->getValue('mail_reply_to'))
      ->set('events', $form_state->getValue('events'))
      ->set('mail_message_ids_header_overwrite_header', $form_state->getValue('mail_message_ids_header_overwrite_header'))
      ->set('created_roles_mode', $form_state->getValue('created_roles_mode'))
      ->set('created_roles', $form_state->getValue('created_roles'))
      ->set('updated_roles_mode', $form_state->getValue('updated_roles_mode'))
      ->set('updated_roles', $form_state->getValue('updated_roles'))
      ->set('deleted_roles_mode', $form_state->getValue('deleted_roles_mode'))
      ->set('deleted_roles', $form_state->getValue('deleted_roles'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
